/*
 * Copyright (c) OSGi Alliance (2014, 2015). All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.osgi.service.enocean;

/**
 * Holds the necessary methods to interact with an EnOcean message.
 * 
 * @author $Id: b60697d5313b69cf5d81174e2dae16953d95244d $
 */
public interface EnOceanMessage {

    /**
     * @return the message's RORG
     */
    public int getRorg();

    /**
     * @return the message's FUNC
     */
    public int getFunc();

    /**
     * @return the message's TYPE
     */
    public int getType();

    /**
     * @return the message's Sender ID
     */
    public int getSenderId();

    /**
     * @return the message's destination ID, or -1
     */
    public int getDestinationId();

    /**
     * Gets the bytes corresponding to the whole message, including the CRC. The
     * generated byte[] array may be sent to an EnOcean gateway and is conform
     * to EnOcean Radio Protocol.
     * 
     * @return The serialized byte list corresponding to the binary message.
     */
    public byte[] getBytes();

    /**
     * Returns the payload bytes of this message.
     * 
     * @return corresponding value.
     */
    public byte[] getPayloadBytes();

    /**
     * Gets the current EnOcean status of the Message. The 'status' byte is
     * actually a bitfield that mainly holds repeater information, teach-in
     * status, and more or less information depending on the radiotelegram type.
     * 
     * 
     * @return the current EnOcean status of this message.
     */
    public int getStatus();

    /**
     * Returns the number of subtelegrams (usually 1) this Message carries.
     * 
     * @return The number of subtelegrams in the case of multiframe messages.
     */
    public int getSubTelNum();

    /**
     * Returns the average RSSI on all the received subtelegrams, including
     * redundant ones.
     * 
     * @return The average RSSI perceived.
     */
    public int getDbm();

    /**
     * Returns the security level of this message, as specified in the 'Security
     * of EnOcean Radio Networks' draft, section 4.2.1.3.
     * 
     * @return The security level format.
     */
    public int getSecurityLevelFormat();

}
