/**
 * OW2 Util
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@objectweb.org
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
* USA
*
* --------------------------------------------------------------------------
* $Id: OSGiDeployer.java 4081 2008-10-05 18:14:02Z benoitf $
* --------------------------------------------------------------------------
*/

package org.ow2.util.ee.deploy.impl.osgi;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployable.OSGiDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployer.IDeployer;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * OSGi deployer that is able to deploy an OSGi Deployable
 * @author Florent BENOIT
 */
public class OSGiDeployer implements IDeployer {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(OSGiDeployer.class);

    /**
     * Bundle Context to use for the deployment.
     */
    private BundleContext bundleContext = null;

    /**
     * List of Bundles that have been installed.
     */
    private Map<String, Bundle> installedBundles = null;


    /**
     * Default constructor.
     */
    public OSGiDeployer() {
        this.installedBundles = new HashMap<String, Bundle>();
    }

    /**
     * Undeploy the given Bundle.
     * @param osgiDeployable the deployable to remove.
     * @throws DeployerException if the bundle is not undeployed.
     */
    protected void undeployBundle(final OSGiDeployable osgiDeployable) throws DeployerException {
        logger.debug("Undeploying {0}", osgiDeployable);

        // Bundle that will be handled
        Bundle osgiBundle = null;

        // Check if it has been installed ?
        if (!isDeployed(osgiDeployable)) {
            throw new DeployerException("Cannot undeploy the bundle '" + osgiDeployable + "' as it was not deployed.");
        }

        // get the bundle Object
        osgiBundle = installedBundles.get(getURL(osgiDeployable).toExternalForm());

        // Check found
        if (osgiBundle == null) {
            throw new DeployerException("No bundle found for the '" + osgiDeployable + "'.");
        }

        // Stop the bundle
        try {
            osgiBundle.stop();
        } catch (BundleException e) {
            throw new DeployerException("Cannot stop the bundle '" + osgiDeployable + "'.", e);
        }

        // Uninstall the bundle
        try {
            osgiBundle.uninstall();
        } catch (BundleException e) {
            throw new DeployerException("Cannot uninstall the bundle '" + osgiDeployable + "'.", e);
        }

        // Remove it of installed bundle
        installedBundles.remove(getURL(osgiDeployable).toExternalForm());

    }


    /**
     * Deploy the given Bundle.
     * @param osgiDeployable the deployable to add.
     * @throws DeployerException if the Bundle is not deployed.
     */
    protected void deployBundle(final OSGiDeployable osgiDeployable) throws DeployerException {
        logger.debug("Deploying {0}", osgiDeployable);

        // Bundle that will be handled
        Bundle osgiBundle = null;

        // First, install the bundle
        try {
            osgiBundle = bundleContext.installBundle("reference:" + getURL(osgiDeployable).toExternalForm());
        } catch (BundleException e) {
            throw new DeployerException("Cannot install the bundle '" + osgiDeployable + "'.", e);
        }

        if (osgiDeployable.isStart()) {
            // And then start it
            try {
                osgiBundle.start();
            } catch (BundleException e) {
                throw new DeployerException("Cannot start the bundle '" + osgiDeployable + "'.", e);
            }
        }

        // Add it as installed
        installedBundles.put(getURL(osgiDeployable).toExternalForm(), osgiBundle);

    }



    /**
     * Deploy an OSGi deployable.
     * @param deployable a given deployable
     * @throws DeployerException if the deployment is not done.
     */
    public void deploy(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Deploy the OSGi Deployable
        if (deployable instanceof OSGiDeployable) {
            deployBundle((OSGiDeployable) deployable);
        }
    }

    /**
     * Checks if the given deployable is deployed or not.
     * @param deployable test if a given deployable is already deployed.
     * @return true if the deployable is deployed.
     * @throws DeployerException if the undeploy operation fails.
     */
    public boolean isDeployed(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Check if Bundle has been deployed
        return installedBundles.get(getURL(deployable).toExternalForm()) != null;
    }


    /**
     * Allows to get the URL from the given Deployable.
     * @param deployable the given deployable.
     * @return an URL object of this deployable
     * @throws DeployerException if the URL can't be obtained.
     */
    protected URL getURL(final IDeployable<?> deployable) throws DeployerException {
        // Get URL
        URL url = null;
        try {
            url = deployable.getArchive().getURL();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get URL from deployable '" + deployable + "'.", e);
        }

        // Get URL
        return url;
    }


    /**
     * @return the bundle context that is used to manage bundles on the gateway.
     */
    public BundleContext getBundleContext() {
        return bundleContext;
    }

    /**
     * Sets the bundle context that is used to manage the bundles on the gateway.
     * @param bundleContext the given bundle context
     */
    public void setBundleContext(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }


    /**
     * Checks if the given deployable is supported by the Deployer.
     * @param deployable the deployable to be checked
     * @return true if it is supported, else false.
     */
    public boolean supports(final IDeployable<?> deployable) {
        return (deployable instanceof OSGiDeployable);
    }

    /**
     * Undeploy the given Bundle deployable.
     * @param deployable a given deployable to undeploy
     * @throws DeployerException if the undeploy operation fails.
     */
    public void undeploy(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Undeploy the OSGi Deployable
        if (deployable instanceof OSGiDeployable) {
            undeployBundle((OSGiDeployable) deployable);
        }
    }

    /**
     * Check if the given deployable is deployable or not.
     * @param deployable the deployable to check.
     * @throws DeployerException if the deployable is not supported.
     */
    private void check(final IDeployable<?> deployable) throws DeployerException {
        if (bundleContext == null) {
            throw new DeployerException("The deployment of the deployable'" + deployable
                    + "' is not available as there is no bundle context set on this deployer.");
        }

        if (!supports(deployable)) {
            throw new DeployerException("The deployment of the deployable'" + deployable
                    + "' is not supported by this deployer.");
        }
    }

}
