/**
 * OW2 Util
 * Copyright (C) 2008 SERLI
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id UrlFetcherActivator.java $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.plan.fetcher.impl.url;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.plan.fetcher.api.IResourceFetcherFactory;
import org.ow2.util.plan.repository.api.IRepositoryManager;


/**
 * Activator for the fetcher/url bundle.
 * It registers a UrlResourceFetcherFactory, registers it as listener for
 * url protocol registration/unregistration.
 * @author Mickaël LEDUQUE
 *
 */
public class UrlFetcherActivator implements BundleActivator, ServiceListener {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(UrlFetcherActivator.class);

    /**
     * The fetcher factory manager.
     */
    UrlResourceFetcherFactory fetcherFactory = null;

    /**
     * The ServiceRegistration object for the registered XmlHelper object.
     */
    private ServiceRegistration urlFetcherFactoryServiceRegistration = null;

    /**
     * The listener filter for this bundle.
     */
    private final static String LISTENER_FILTER_STRING = 
        "(" + Constants.OBJECTCLASS + "=" + IRepositoryManager.class.getName() + ")";

    /**
     * The bundle context.
     */
    private BundleContext bundleContext = null;

    /**
     * {@inheritDoc}
     * 
     * Specific : sets up and registers a Url resource fetcher.
     */
    public void start(BundleContext context) throws Exception {
        this.bundleContext = context;
        fetcherFactory = new UrlResourceFetcherFactory();
        /* registers self as service events listener */
        context.addServiceListener(this,LISTENER_FILTER_STRING);

        /* Set up the repository manager */
        ServiceReference repositoryManagerRef = context.getServiceReference(
                IRepositoryManager.class.getName());
        if (repositoryManagerRef != null) {
            IRepositoryManager repositoryManager = (IRepositoryManager)context.getService(repositoryManagerRef);
            fetcherFactory.setRepositoryManager(repositoryManager);
            logger.debug("the repository manager is set to " + repositoryManager);
        }
        /* registers the factory as both fetcher factory and protocol manager */
        this.urlFetcherFactoryServiceRegistration = context.registerService(
                        IResourceFetcherFactory.class.getName(),
                        fetcherFactory, null);

    }

    /**
     * {@inheritDoc}
     */
    public void stop(BundleContext context) throws Exception {
        urlFetcherFactoryServiceRegistration.unregister();
        logger.debug("URL fetcher factory unregistered.");
    }

    /**
     * {@inheritDoc}
     * 
     * Specific : listens to repository manager registration
     * and unregistration events.
     */
    public void serviceChanged(ServiceEvent event) {
        /* Will process fetcher factories registration and unregistration events
         * to maintain factory list */
        ServiceReference serviceRef = event.getServiceReference();
        Object serviceObject = bundleContext.getService(serviceRef);
        if (serviceObject instanceof IRepositoryManager) {
            IRepositoryManager repositoryManager = (IRepositoryManager) serviceObject;
            switch (event.getType()) {
                case ServiceEvent.REGISTERED :
                    if (fetcherFactory.getRepositoryManager() == null) {
                        fetcherFactory.setRepositoryManager(repositoryManager);
                        logger.debug("The repository manager is set to '{0}'.", repositoryManager);
                    }
                    break;
                case ServiceEvent.UNREGISTERING :
                    fetcherFactory.setRepositoryManager(null);
                    logger.debug("The repository manager is unset.");
                    break;
                default : break;
            }
        }
    }

}
