/**
 * OW2 Util
 * Copyright (C) 2008 SERLI
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id FetcherActivator.java $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.plan.fetcher.impl;

import java.util.Dictionary;
import java.util.Hashtable;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.ManagedService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.plan.fetcher.api.IResourceFetcherFactory;
import org.ow2.util.plan.fetcher.api.IResourceFetcherFactoryManager;

/**
 * Activator for the fetcher/impl bundle. It registers a
 * ResourceFetcherFactoryManager service and uses it as
 * registration/unregistration event listener for ResourceFetcherFactory
 * services.
 * @author Mickaël LEDUQUE
 */
public class FetcherActivator implements BundleActivator, ServiceListener {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(FetcherActivator.class);

    /**
     * The fetcher factory manager.
     */
    ResourceFetcherFactoryManagerImpl fetcherManager = null;

    /**
     * The ServiceRegistration object for the registered XmlHelper object.
     */
    private ServiceRegistration fetcherManagerServiceRegistration = null;

    /**
     * The bundle context.
     */
    private BundleContext bundleContext = null;

    /**
     * The LocalDirectoryConfiguration reference.
     */
    private ServiceRegistration configRegistration = null;

    /**
     * {@inheritDoc}
     */
    public void start(BundleContext context) throws Exception {
        this.bundleContext = context;
        fetcherManager = new ResourceFetcherFactoryManagerImpl();
        /* registers self as service events listener */
        StringBuilder sb = new StringBuilder("(");
        sb.append(Constants.OBJECTCLASS);
        sb.append("=");
        sb.append(IResourceFetcherFactory.class.getName());
        sb.append(")");
        context.addServiceListener(this, sb.toString());
        /* registers the factory manager */
        this.fetcherManagerServiceRegistration = context.registerService(IResourceFetcherFactoryManager.class.getName(),
                fetcherManager, null);
        /*
         * try to get the factories that were registered before this was
         * registered as listener
         */
        ServiceReference[] preFactories = context.getServiceReferences(IResourceFetcherFactory.class.getName(), null);
        if (preFactories != null) { // getServiceReferences return null instead
            // of empty array
            for (ServiceReference ref : preFactories) {
                IResourceFetcherFactory fetcherFactory = (IResourceFetcherFactory) context.getService(ref);
                fetcherManager.addResourceFetcherFactory(fetcherFactory);
                logger.debug("Found one resource fetcher factory for {0} resources", fetcherFactory.getSupportedClass());
            }
        }

        // Create and register the LocalRepositoriesBaseDirConfiguration as
        // ManagedServiceFactory
        LocalRepositoriesBaseDirConfiguration config = new LocalRepositoriesBaseDirConfiguration(this.fetcherManager);
        Dictionary<String, String> props = new Hashtable<String, String>();
        props.put(Constants.SERVICE_PID, org.ow2.util.plan.fetcher.api.Constants.CONFIG_PID);
        this.configRegistration = context.registerService(ManagedService.class.getName(), config, props);
    }

    /**
     * {@inheritDoc}
     */
    public void stop(BundleContext context) throws Exception {
        this.configRegistration.unregister();
        this.fetcherManagerServiceRegistration.unregister();
    }

    /**
     * {@inheritDoc}
     */
    public void serviceChanged(ServiceEvent event) {
        /*
         * Will process fetcher factories registration and unregistration events
         * to maintain factory list
         */
        ServiceReference serviceRef = event.getServiceReference();
        /* we are registered as a listener with a filter on service class */
        IResourceFetcherFactory factory = (IResourceFetcherFactory) bundleContext.getService(serviceRef);
        switch (event.getType()) {
        case ServiceEvent.REGISTERED:
            fetcherManager.addResourceFetcherFactory(factory);
            logger.debug("Added one resource fetcher factory for {0} resources", factory.getSupportedClass());
            break;
        case ServiceEvent.UNREGISTERING:
            fetcherManager.removeResourceFetcherFactory(factory);
            logger.debug("Removed one resource fetcher factory for {0} resources", factory.getSupportedClass());
            break;
        default:
            break;
        }
    }

}
