/**
 * OW2 Util
 * Copyright (C) 2008 SERLI
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id ReaderActivator.java $
 * --------------------------------------------------------------------------
 */

package org.ow2.util.plan.reader.activator;

import java.util.Dictionary;
import java.util.Hashtable;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.ServiceListener;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.plan.bindings.schemastream.ISchemaStreamFactory;
import org.ow2.util.plan.reader.ReaderException;
import org.ow2.util.plan.reader.plan.IPlanReader;
import org.ow2.util.plan.reader.plan.PlanReaderImpl;
import org.ow2.util.plan.reader.repository.IRepositoryDataReader;
import org.ow2.util.plan.reader.repository.RepositoryDataReaderImpl;


/**
 * Activator for the reader bundle.
 * @author mleduque
 */
public class ReaderActivator implements BundleActivator, ServiceListener {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(ReaderActivator.class);

    /**
     * The bundle context.
     */
    private BundleContext bundleContext = null;

    /**
     * The schema InputStream factory.
     */
    private ISchemaStreamFactory schemaStreamFactory = null;

    /**
     * The non-validating plan reader.
     */
    private IPlanReader nonValidatingPlanReader = null;


    /**
     * The non-validating repository reader.
     */
    private IRepositoryDataReader nonValidatingRepositoryReader = null;


    /**
     * The validating plan reader.
     */
    private IPlanReader validatingPlanReader = null;


    /**
     * The validating repository reader.
     */
    private IRepositoryDataReader validatingRepositoryReader = null;

    /**
     * The ServiceRegistration object for the registered PlanReader object.
     */
    private ServiceRegistration nonValidatingPlanReaderServiceRegistration = null;

    /**
     * The ServiceRegistration object for the registered PlanReader object.
     */
    private ServiceRegistration nonValidatingRepositoryReaderServiceRegistration = null;

    /**
     * The validating plan reader registration object.
     */
    private ServiceRegistration validatingPlanReaderServiceRegistration = null;

    /**
     * The validating repository reader registration object.
     */
    private ServiceRegistration validatingRepositoryReaderServiceRegistration = null;

    /**
     * {@inheritDoc}
     */
    public void start(BundleContext context) throws Exception {
        this.bundleContext = context;

        // TODO I though I had this removed...
        Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());

        this.registerNonValidatingReaders();

        ServiceReference schemaFactoryRef = context.getServiceReference(
                                            ISchemaStreamFactory.class.getName());
        if (schemaFactoryRef != null) {
            this.schemaStreamFactory = (ISchemaStreamFactory)context.getService(schemaFactoryRef);
            this.registerValidatingReaders();
        }

        this.bundleContext.addServiceListener(this, "(" + Constants.OBJECTCLASS + "=" +
                ISchemaStreamFactory.class.getName() + ")");
    }

    /**
     * Registers the non-validating readers.
     */
    private void registerNonValidatingReaders() throws Exception {
        this.nonValidatingPlanReader = new PlanReaderImpl();
        this.nonValidatingRepositoryReader = new RepositoryDataReaderImpl();
        Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put("validating", Boolean.FALSE);
        this.nonValidatingPlanReaderServiceRegistration = this.bundleContext.registerService(
                IPlanReader.class.getName(), nonValidatingPlanReader, properties);
        this.nonValidatingRepositoryReaderServiceRegistration = this.bundleContext.registerService(
                IRepositoryDataReader.class.getName(), nonValidatingRepositoryReader, properties);
        logger.debug("Non validating Plan Reader {0} registered - Non validating Repository Reader {1} registered.",
                nonValidatingPlanReader, nonValidatingRepositoryReader);
    }

    /**
     * Unregisters the non-validating readers.
     */
    private void unRegisterNonValidatingReaders() throws Exception {
        Exception firstUnregisterException = null;
        if (nonValidatingPlanReaderServiceRegistration != null) {
            try {
                nonValidatingPlanReaderServiceRegistration.unregister();
                logger.debug("Plan Reader unregistered");
            } catch (Exception e) {
                firstUnregisterException = e;
            } finally {
                nonValidatingPlanReaderServiceRegistration = null;
            }
        }
        if (nonValidatingRepositoryReaderServiceRegistration != null) {
            nonValidatingRepositoryReaderServiceRegistration.unregister();
            logger.debug("Repository Reader unregistered");
            nonValidatingRepositoryReaderServiceRegistration = null;
        }

        if (firstUnregisterException != null) {
            throw new Exception(firstUnregisterException);
        }
    }

    /**
     * Tries to register the validating readers.
     */
    private void registerValidatingReaders() {
        try {
            this.validatingPlanReader = new PlanReaderImpl();
            this.validatingRepositoryReader = new RepositoryDataReaderImpl();
        } catch (Exception e) {
            logger.warn("Couldn't create the validating readers ; exception {0}", e);
            this.validatingPlanReader = null;
            this.validatingRepositoryReader = null;
            return;
        }

        this.validatingPlanReader.setSchemaStreamFactory(this.schemaStreamFactory);
        this.validatingRepositoryReader.setSchemaStreamFactory(this.schemaStreamFactory);

        try {
            this.validatingPlanReader.setValidation(true);
            this.validatingRepositoryReader.setValidation(true);
        } catch (ReaderException e) {
            logger.warn("Couldn't activate readers validation ; exception {0}", e);
            this.validatingPlanReader = null;
            this.validatingRepositoryReader = null;
            return;
        }

        Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put("validating", Boolean.TRUE);
        this.validatingPlanReaderServiceRegistration = this.bundleContext.registerService(
                IPlanReader.class.getName(), this.validatingPlanReader, properties);
        this.validatingRepositoryReaderServiceRegistration = this.bundleContext.registerService(
                IRepositoryDataReader.class.getName(), this.validatingRepositoryReader, properties);
        logger.debug("Validating Plan Reader {0} registered - Validating Repository Reader {1} registered.",
                validatingPlanReader, validatingRepositoryReader);
    }

    /**
     * Unregisters the validating readers.
     */
    private void unregisterValidatingReaders() {
        if (this.validatingPlanReaderServiceRegistration != null) {
            this.validatingPlanReaderServiceRegistration.unregister();
            logger.debug("Validating Plan Reader unregistered");
            this.validatingPlanReaderServiceRegistration = null;
        }
        if (this.validatingRepositoryReaderServiceRegistration != null) {
            this.validatingRepositoryReaderServiceRegistration.unregister();
            logger.debug("Validating Repository Reader unregistered");
            this.validatingRepositoryReaderServiceRegistration = null;
        }
    }

    /**
     * {@inheritDoc}
     */
    public void stop(BundleContext context) throws Exception {
        this.unregisterValidatingReaders();
        this.unRegisterNonValidatingReaders();
    }

    /**
     * {@inheritDoc}
     */
    public void serviceChanged(ServiceEvent event) {
        ServiceReference serviceRef = event.getServiceReference();
        Object service = this.bundleContext.getService(serviceRef);
        if (service instanceof ISchemaStreamFactory) {
            switch (event.getType()) {
            case ServiceEvent.REGISTERED :
                if (this.validatingPlanReaderServiceRegistration != null) {
                    this.registerValidatingReaders();
                }
                break;
            case ServiceEvent.UNREGISTERING :
                this.unregisterValidatingReaders();
                break;
            default :
                break;

            }
        }

    }

}
