/**
 * Copyright (C) 2010 Bull S.A.S.
 *
 * CAROL: Common Architecture for RMI ObjectWeb Layer
 *
 * This library is developed inside the ObjectWeb Consortium,
 * http://www.ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Activator.java 2215 2011-09-22 15:45:11Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.osgi.internal;

import java.util.Dictionary;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Set;

import javax.naming.spi.InitialContextFactory;
import javax.naming.spi.ObjectFactory;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.log.LogService;
import org.ow2.carol.jndi.enc.java.javaURLContextFactory;
import org.ow2.carol.jndi.intercept.spi.InterceptorInitialContextFactory;
import org.ow2.carol.jndi.spi.IIOPContextWrapperFactory;
import org.ow2.carol.jndi.spi.IRMIContextWrapperFactory;
import org.ow2.carol.jndi.spi.JRMPContextWrapperFactory;
import org.ow2.carol.jndi.spi.JacORBIIOPContextWrapperFactory;
import org.ow2.carol.jndi.spi.LMIContextWrapperFactory;
import org.ow2.carol.jndi.spi.MultiOrbInitialContextFactory;
import org.ow2.carol.jndi.spi.URLInitialContextFactory;

/**
 * A {@code Activator} is ...
 *
 * @author Guillaume Sauthier
 */
public class Activator implements BundleActivator {

    /**
     * Registrations of the InitialContextFactories provided by Carol.
     */
    private Set<ServiceRegistration> registrations;

    /**
     * Default constructor.
     */
    public Activator() {
        this.registrations = new HashSet<ServiceRegistration>();
    }

    /**
     * Register all the known InitialContextFactories of this bundle.
     * @param context OSGi BundleContext
     * @throws Exception
     */
    public void start(final BundleContext context) throws Exception {

        // Usual ICFs
        registerInitialContextFactory(context, new IIOPContextWrapperFactory());
        registerInitialContextFactory(context, new IRMIContextWrapperFactory());
        registerInitialContextFactory(context, new JacORBIIOPContextWrapperFactory());
        registerInitialContextFactory(context, new JRMPContextWrapperFactory());
        registerInitialContextFactory(context, new LMIContextWrapperFactory());
        registerInitialContextFactory(context, new MultiOrbInitialContextFactory());

        // Legacy ICFs
        registerInitialContextFactory(context, new org.objectweb.carol.jndi.spi.MultiOrbInitialContextFactory());

        // Interceptable ICFs
        registerInitialContextFactory(context, new InterceptorInitialContextFactory());

        // A required ICF needed for plumbing ...
        registerInitialContextFactory(context, new URLInitialContextFactory());

        // Carol java:/ URLContextFactory
        try {
            registerURLContextFactory(context, new javaURLContextFactory(), "java");
        } catch (Throwable error) {
            // This could be ignored, for example, in JOnAS, this URL ContextFactory is not used

            String message = "org.ow2.carol.jndi.enc.java.javaURLContextFactory was not found, it will be ignored.";
            LogService logger = findLogService(context);
            if (logger != null) {
                logger.log(LogService.LOG_DEBUG, message, error);
            } else {
                System.err.println("[DEBUG] " + message);
            }
        }

    }

    private LogService findLogService(BundleContext context) {
        ServiceReference ref = context.getServiceReference(LogService.class.getName());
        if (ref != null) {
            return (LogService) context.getService(ref);
        }
        return null;
    }

    /**
     * Retract all previously registered InitialContextFactory services.
     * @param context OSGi BundleContext
     * @throws Exception never thrown
     */
    public void stop(final BundleContext context) throws Exception {

        // Un-register all InitialContextFactories
        for (ServiceRegistration registration : registrations) {
            registration.unregister();
        }

    }

    /**
     * Register the given InitialContextFactory instance with the mandatory names.
     * @param context OSGi BundleContext
     * @param factory service instance to be registered
     */
    private void registerInitialContextFactory(final BundleContext context,
                                               final InitialContextFactory factory) {
        String[] interfaces = new String[] {
                InitialContextFactory.class.getName(),
                factory.getClass().getName()
        };
        registrations.add(context.registerService(interfaces, factory, null));
    }

    /**
     * Register the given ObjectFactory instance with the given schemes.
     * @param context OSGi BundleContext
     * @param urlContextFactory Special ObjectFactory for Context creation from URL
     * @param scheme supported scheme
     * @param factory service instance to be registered
     */
    private void registerURLContextFactory(final BundleContext context,
                                           final ObjectFactory urlContextFactory,
                                           final String scheme) {

        // Store the scheme
        // Maybe store the 'service.ranking' too
        Dictionary<String, Object> props = new Hashtable<String, Object>();
        props.put("osgi.jndi.url.scheme", scheme);
        //props.put("service.ranking", 0);

        // Register
        registrations.add(context.registerService(ObjectFactory.class.getName(),
                                                  urlContextFactory,
                                                  props));
    }

}
