/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:CMIAdmin.java 949 2007-06-02 17:24:33Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.admin;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;
import javax.management.remote.JMXServiceURL;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.controller.common.AbsClusterViewManager;
import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.controller.server.ServerClusterViewManager;
import org.ow2.carol.cmi.controller.server.ServerClusterViewManagerException;
import org.ow2.carol.cmi.lb.PropertyConfigurationException;
import org.ow2.carol.cmi.lb.data.LBPropertyData;
import org.ow2.carol.cmi.lb.policy.ILBPolicy;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.carol.cmi.lb.util.LBPolicyFactory;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


/**
 * Provides methods to admin a cluster. This class is a front end for users of cluster.
 * Low level aspects are delegated at a ClusterViewManager.
 * @author The new CMI team
 */
@ThreadSafe
public final class CMIAdmin implements CMIAdminMBean {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(CMIAdmin.class);

    /**
     * clusterViewManager to delegate tasks.
     */
    private final ClusterViewManager clusterViewManager;

    /**
     * The name binded in the MBean Server.
     */
    private final ObjectName objectName;

    /**
     * Singleton.
     */
    private static CMIAdmin cmiAdmin;

    /**
     * Constructs a CMIAdmin.
     * @param clusterViewManager a ClusterViewManager to delegate tasks
     */
    private CMIAdmin(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
        objectName = null;
    }

    /**
     * Constructs a CMIAdmin MBean.
     * @param serverClusterViewManager a ServerClusterViewManager to delegate tasks
     * @param objectName the name binded in the MBean Server
     */
    private CMIAdmin(final ServerClusterViewManager serverClusterViewManager, final ObjectName objectName) {
        this.clusterViewManager = serverClusterViewManager;
        this.objectName = objectName;
    }


    /**
     * Returns an instance of CMIAdmin MBean.
     * @param objectName the name binded in the MBean Server
     * @param serverClusterViewManager the manager of the cluster view to use
     * @return an instance of CMIAdmin MBean
     */
    public static synchronized CMIAdmin getCMIAdmin(
            final ObjectName objectName, final ServerClusterViewManager serverClusterViewManager) {
        if(cmiAdmin == null) {
            cmiAdmin = new CMIAdmin(serverClusterViewManager, objectName);
        }
        return cmiAdmin;
    }

    /**
     * Returns an instance of CMIAdmin.
     * @return an instance of CMIAdmin
     */
    public static synchronized CMIAdmin getCMIAdmin() {
        if(cmiAdmin == null) {
            ClusterViewManager clusterViewManager = AbsClusterViewManager.getClusterViewManager();
            cmiAdmin = new CMIAdmin(clusterViewManager);
        }
        return cmiAdmin;
    }

    /**
     * Returns the ObjectName binded in the MBean Server.
     * @return the ObjectName binded in the MBean Server
     */
    public ObjectName getObjectName() {
        return objectName;
    }

    /**
     * @param protocolName a name of protocol
     * @return the JMX connector URL to access to this MBean
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public JMXServiceURL getJMXServiceURL(final String protocolName)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        JMXServiceURL jmxServiceURL = ((ServerClusterViewManager) clusterViewManager).getJMXServiceURL(protocolName);
        if(jmxServiceURL == null) {
            throw new IllegalArgumentException("Unknown protocol: "+protocolName);
        }
        return jmxServiceURL;
    }

    /**
     * @return the protocols registered in the manager
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public Set<String> getProtocols() throws UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        return ((ServerClusterViewManager) clusterViewManager).getProtocols();
    }

    /**
     * Returns a name of interface of this object.
     * @param objectName a name of object
     * @return a name of interface of this object
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public String getItfName(final String objectName)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        try {
            return ((ServerClusterViewManager) clusterViewManager).getItfName(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns the business interface name of an object bound with the given name (for ejb2 only).
     * If the object is not an ejb2, null is returned.
     * @param objectName a name of object
     * @return the business interface name of an object bound with the given name
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public String getBusinessName(final String objectName)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        try {
            return ((ServerClusterViewManager) clusterViewManager).getBusinessName(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * @param objectName a name of object
     * @return true if the object with the given name is replicated
     * @throws IllegalArgumentException if none object has the given name
     */
    public boolean isReplicated(String objectName) throws IllegalArgumentException {
        try {
            return clusterViewManager.isReplicated(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * @param protocolName a name of protocol
     * @return the set of references on server connected to this server
     * @throws IllegalArgumentException if the given protocol name doesn't exist
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public Set<String> getServerRefsForProtocol(final String protocolName)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        Set<ServerRef> serverRefs;
        try {
            serverRefs = ((ServerClusterViewManager) clusterViewManager).getServerRefs(protocolName);
        } catch (ServerClusterViewManagerException e) {
            throw new IllegalArgumentException("Protocol not found: "+protocolName+"\n"+e.getMessage());
        }
        Set<String> result = new HashSet<String>();
        for(ServerRef serverRef : serverRefs) {
            result.add(serverRef.getProviderURL());
        }
        return result;
    }

    /**
     * @return the set of name of object
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public Set<String> getObjectNames() throws UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        return ((ServerClusterViewManager) clusterViewManager).getObjectNames();
    }

    /**
     * @param protocolName a name of protocol
     * @return the reference on the local registry for the given protocol
     * @throws IllegalArgumentException if the given protocol name doesn't exist
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public String getRefOnLocalRegistry(final String protocolName)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        ServerRef serverRef = ((ServerClusterViewManager) clusterViewManager).getRefOnLocalRegistry(protocolName);
        if(serverRef == null) {
            throw new IllegalArgumentException(serverRef+" doesn't exist");
        }
        return serverRef.getProviderURL();
    }

    /**
     * Returns the name of class of policy for the object with the given name.
     * @param objectName name of the object
     * @return the name of class of policy for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    public String getLBPolicyClassName(final String objectName) throws IllegalArgumentException {
        try {
            return clusterViewManager.getLBPolicyClassName(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Sets a new policy for a given object.
     * @param objectName a name of object
     * @param lbPolicyClassName a name of class of LB policy
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws ClassNotFoundException if the class is missing
     */
    @SuppressWarnings("unchecked")
    public void setLBPolicyClassName(final String objectName, final String lbPolicyClassName)
    throws IllegalArgumentException, UnsupportedOperationException, ClassNotFoundException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        Class<?> lbPolicyClass = Class.forName(lbPolicyClassName);
        Class[] interfaces = lbPolicyClass.getInterfaces();
        if(!Arrays.asList(interfaces).contains(ILBPolicy.class)) {
            throw new IllegalArgumentException(lbPolicyClassName+" doesn't implement the interface "+ILBPolicy.class.getName());
        }
        try {
            ((ServerClusterViewManager) clusterViewManager).setLBPolicyClassName(objectName, lbPolicyClassName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns the name of class of strategy for the object with the given name.
     * @param objectName name of the object
     * @return the name of class of strategy for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    public String getLBStrategyClassName(final String objectName)
    throws IllegalArgumentException {
        try {
            return clusterViewManager.getLBStrategyClassName(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Sets a new strategy for a given object.
     * @param objectName a name of object
     * @param lbStrategyClassName a name of class of LB strategy
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws ClassNotFoundException if the class is missing
     */
    @SuppressWarnings("unchecked")
    public void setLBStrategyClassName(final String objectName, final String lbStrategyClassName)
    throws IllegalArgumentException, UnsupportedOperationException, ClassNotFoundException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        Class<?> lbStrategyClass = Class.forName(lbStrategyClassName);
        Class[] interfaces = lbStrategyClass.getInterfaces();
        if(!Arrays.asList(interfaces).contains(ILBStrategy.class)) {
            throw new IllegalArgumentException(
                    lbStrategyClassName+" doesn't implement the interface "+ILBStrategy.class.getName());
        }
        try {
            ((ServerClusterViewManager) clusterViewManager).setLBStrategyClassName(objectName, lbStrategyClassName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns the set of property names for the object with the given name.
     * @param objectName a name of object
     * @return the set of property names for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    public Set<String> getPropertiesNamesForLBPolicy(final String objectName) throws IllegalArgumentException {
        Map<String, LBPropertyData> properties;
        String lbPolicyClassname;
        try {
            lbPolicyClassname = clusterViewManager.getLBPolicyClassName(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
        properties = LBPolicyFactory.getPropertyData(lbPolicyClassname);
        Set<String> propertiesNames = new HashSet<String>();
        for(String propertyName : properties.keySet()) {
            if(!List.class.equals(
                    LBPolicyFactory.getPropertyRawType(lbPolicyClassname, propertyName))) {
                propertiesNames.add(propertyName);
            }
        }
        return propertiesNames;
    }

    /**
     * Returns the set of property names (for which value is a list) for the object with the given name.
     * @param objectName a name of object
     * @return the set of property names for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    public Set<String> getListPropertiesNamesForLBPolicy(final String objectName) throws IllegalArgumentException {
        Map<String, LBPropertyData> properties;
        String lbPolicyClassname;
        try {
            lbPolicyClassname = clusterViewManager.getLBPolicyClassName(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
        properties = LBPolicyFactory.getPropertyData(lbPolicyClassname);
        Set<String> propertiesNames = new HashSet<String>();
        for(String propertyName : properties.keySet()) {
            if(List.class.equals(
                    LBPolicyFactory.getPropertyRawType(lbPolicyClassname, propertyName))) {
                propertiesNames.add(propertyName);
            }
        }
        return propertiesNames;
    }

    /**
     * Returns the value of the property with the given name.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @return the value of the property with the given name, or null if there is not property for this name
     * @throws IllegalArgumentException if none object has the given name, or if the value is a list
     */
    public String getPropertyForLBPolicy(final String objectName, final String propertyName)
    throws IllegalArgumentException {
        ILBPolicy<CMIReference> lbPolicy;
        try {
            lbPolicy = clusterViewManager.getLBPolicy(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
        Type propertyType = LBPolicyFactory.getPropertyRawType(lbPolicy.getClass().getName(), propertyName);
        if(List.class.equals(propertyType)) {
            LOGGER.error("The value of property {0} is a list", objectName);
            throw new IllegalArgumentException("The value of property "+objectName+" is a list");
        }
        return LBPolicyFactory.getProperty(lbPolicy, propertyName).toString();
    }

    /**
     * Returns the list of value of the property with the given name.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @return the list of value of the property with the given name, or null if there is not property for this name
     * @throws IllegalArgumentException if none object has the given name, or if if the value is not a list
     */
    public List<String> getListPropertyForLBPolicy(final String objectName, final String propertyName)
    throws IllegalArgumentException {
        ILBPolicy<CMIReference> lbPolicy;
        try {
            lbPolicy = clusterViewManager.getLBPolicy(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
        Type propertyType = LBPolicyFactory.getPropertyRawType(lbPolicy.getClass().getName(), propertyName);
        if(!List.class.equals(propertyType)) {
            LOGGER.error("The value of property {0} is not a list", objectName);
            throw new IllegalArgumentException("The value of property "+objectName+" is not a list");
        }
        List<String> result = new ArrayList<String>();

        for(Object obj : (List<?>) LBPolicyFactory.getProperty(lbPolicy, propertyName)) {
            result.add(obj.toString());
        }
        return result;
    }

    /**
     * Sets a property for a given object.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @param propertyValue a value for the given name of property
     * @throws IllegalArgumentException if none object has the given name, or if if the property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    @SuppressWarnings("unchecked")
    public void setPropertyForLBPolicy(final String objectName, final String propertyName, final String propertyValue)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        try {
            Class<? extends ILBPolicy> lbPolicyClass = clusterViewManager.getLBPolicyClass(objectName);
            ((ServerClusterViewManager) clusterViewManager).setPropertyForLBPolicy(
                    objectName, propertyName, LBPolicyFactory.convertString(lbPolicyClass, propertyName, propertyValue));
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        } catch (PropertyConfigurationException e) {
            throw new IllegalArgumentException("The property "+propertyName+" cannot be applied", e);
        }
    }

    /**
     * Sets a property (a list of value) for a given object.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @param propertyValue a list of value for the given name of property
     * @throws IllegalArgumentException if none object has the given name, or if the property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    @SuppressWarnings("unchecked")
    public void setListPropertyForLBPolicy(final String objectName, final String propertyName, final List<String> propertyValue)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        try {
            Class<? extends ILBPolicy> lbPolicyClass = clusterViewManager.getLBPolicyClass(objectName);
            ((ServerClusterViewManager) clusterViewManager).setPropertyForLBPolicy(
                    objectName, propertyName, LBPolicyFactory.convertStrings(lbPolicyClass, propertyName, propertyValue));
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        } catch (PropertyConfigurationException e) {
            throw new IllegalArgumentException("The property "+propertyName+" cannot be applied"+"\n"+e.getMessage());
        }
    }

    /**
     * Sets the properties for a given object. A property is either a String or a list of String.
     * @param objectName a name of object
     * @param properties properties a set of properties
     * @throws IllegalArgumentException if none object has the given name, or if a property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    @SuppressWarnings("unchecked")
    public void setPropertiesForLBPolicy(final String objectName, final Map<String, Object> properties)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        try {
            Class<? extends ILBPolicy> lbPolicyClass = clusterViewManager.getLBPolicyClass(objectName);
            ((ServerClusterViewManager) clusterViewManager).setPropertiesForLBPolicy(
                    objectName, convertProperties(lbPolicyClass, properties));
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        } catch (PropertyConfigurationException e) {
            throw new IllegalArgumentException("The properties "+properties+" cannot be applied"+"\n"+e.getMessage());
        }
    }

    /**
     * Sets the algorithm of load-balancing for the object with the given name.
     * @param objectName a name of object
     * @param lbPolicyClassName a name of class of LB policy
     * @param lbStrategyClassName a name of class of LB strategy
     * @param properties a set of properties
     * @throws IllegalArgumentException if none object has the given name, or if a property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws ClassNotFoundException if the class is missing
     */
    @SuppressWarnings("unchecked")
    public void setAlgorithmForLBPolicy(final String objectName,
            final String lbPolicyClassName, final String lbStrategyClassName, final Map<String, Object> properties)
    throws IllegalArgumentException, UnsupportedOperationException, ClassNotFoundException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        Class<?> lbPolicyClass = Class.forName(lbPolicyClassName);
        Class[] interfaces = lbPolicyClass.getInterfaces();
        if(!Arrays.asList(interfaces).contains(ILBPolicy.class)) {
            throw new IllegalArgumentException(lbPolicyClassName+" doesn't implement the interface "+ILBPolicy.class.getName());
        }
        Class<?> lbStrategyClass = Class.forName(lbStrategyClassName);
        interfaces = lbStrategyClass.getInterfaces();
        if(!Arrays.asList(interfaces).contains(ILBStrategy.class)) {
            throw new IllegalArgumentException(
                    lbStrategyClassName+" doesn't implement the interface "+ILBStrategy.class.getName());
        }
        try {
            ((ServerClusterViewManager) clusterViewManager).setAlgorithmForLBPolicy(
                    objectName, lbPolicyClassName, lbStrategyClassName,
                    convertProperties((Class<? extends ILBPolicy>) lbPolicyClass, properties));
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        } catch (PropertyConfigurationException e) {
            throw new IllegalArgumentException("The properties "+properties+" cannot be applied"+"\n"+e.getMessage());
        }
    }

    /**
     * @param lbPolicyClass a class implementing a policy
     * @param properties properties
     * @return values with the good type
     * @throws ObjectNotFoundException if no object has the given name
     * @throws PropertyConfigurationException if a property doesn't exist or has an invalid type
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> convertProperties(
            final Class<? extends ILBPolicy> lbPolicyClass, final Map<String, Object> properties)
            throws ObjectNotFoundException, PropertyConfigurationException {
        Map<String, Object> castedProps = new HashMap<String, Object>();
        for(String propertyName : properties.keySet()) {
            Type propertyType = LBPolicyFactory.getPropertyType(lbPolicyClass.getName(), propertyName);
            Object value = properties.get(propertyName);
            if(propertyType instanceof ParameterizedType) {
                ParameterizedType parameterizedType = (ParameterizedType) propertyType;
                if(parameterizedType.getRawType().equals(List.class)) {
                    if(parameterizedType.getActualTypeArguments().equals(String.class)) {
                        value = LBPolicyFactory.convertStrings(
                                lbPolicyClass, propertyName, (List<String>) properties.get(propertyName));
                    }
                } else {
                    throw new PropertyConfigurationException("Support only list as parameterized type");
                }
            } else {
                if(propertyType.equals(String.class)) {
                    value = LBPolicyFactory.convertString(
                            lbPolicyClass, propertyName, (String) properties.get(propertyName));
                }
            }
            castedProps.put(propertyName, value);
        }
        return castedProps;
    }

    /**
     * Returns a list of String representing a ServerRef for an object with the given name and protocol.
     * @param objectName a name of object
     * @param protocolName a name of protocol
     * @return a list of String representing a ServerRef for an object with the given name and protocol
     * @throws IllegalArgumentException if none object has the given name
     */
    public List<String> getServerRefs(final String objectName, final String protocolName)
    throws IllegalArgumentException {
        List<CMIReference> cmiReferences;
        try {
            cmiReferences = clusterViewManager.getCMIReferences(objectName, protocolName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
        List<String> stringOfCMIRefs = new ArrayList<String>(cmiReferences.size());
        for(CMIReference cmiReference : cmiReferences) {
            stringOfCMIRefs.add(cmiReference.getServerRef().getProviderURL());
        }
        return stringOfCMIRefs;
    }

    /**
     * Returns a list of String representing a ServerRef for an object with the given name.
     * @param objectName a name of object
     * @return a list of String representing a ServerRef for an object with the given name
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public List<String> getServerRefs(final String objectName)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        List<CMIReference> cmiReferences;
        try {
            cmiReferences = ((ServerClusterViewManager) clusterViewManager).getCMIReferences(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
        List<String> stringOfCMIRefs = new ArrayList<String>(cmiReferences.size());
        for(CMIReference cmiReference : cmiReferences) {
            stringOfCMIRefs.add(cmiReference.getServerRef().getProviderURL());
        }
        return stringOfCMIRefs;
    }

    /**
     * Returns the set of name of cluster.
     * @return the set of name of cluster
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public Set<String> getClusterNames() throws UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        Set<String> clusterNames = ((ServerClusterViewManager) clusterViewManager).getClusterNames();
        return clusterNames;
    }

    /**
     * @param clusterName The cluster name
     * @return The set of object names included in the given cluster
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public Set<String> getObjectNames(final String clusterName) throws UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        Set<String> objectNames = ((ServerClusterViewManager) clusterViewManager).getObjectNames(clusterName);
        return objectNames;
    }

    /**
     * Returns the time between each update of the cluster view by clients.
     * @return the time between each update of the cluster view by clients
     */
    public Integer getDelayToRefresh() {
        return clusterViewManager.getDelayToRefresh();
    }

    /**
     * Sets the time between each update of the cluster view by clients.
     * @param delay the time between each update of the cluster view by clients
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public void setDelayToRefresh(final Integer delay) throws UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        ((ServerClusterViewManager) clusterViewManager).setDelayToRefresh(delay);
    }

    /**
     * Returns the name of cluster for the object with the given name.
     * @param objectName a name of object
     * @return the name of cluster for a object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    public String getClusterName(final String objectName) throws IllegalArgumentException {
        try {
            return clusterViewManager.getClusterName(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns the maximal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the maximal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    public Integer getMaxPoolSize(final String objectName) throws IllegalArgumentException {
        try {
            return clusterViewManager.getMaxPoolSize(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns the minimal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the minimal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    public Integer getMinPoolSize(final String objectName) throws IllegalArgumentException {
        try {
            return clusterViewManager.getMinPoolSize(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns true if the pool for object with the given name should be empty.
     * @param objectName a name of object
     * @return true if the pool for object with the given name should be empty
     * @throws IllegalArgumentException if no object is bound with the given name
     */
    public boolean isPoolToEmpty(final String objectName) throws IllegalArgumentException {
        try {
            return clusterViewManager.isPoolToEmpty(objectName);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns true the server with the given reference if blacklisted.
     * @param serverName a reference on a server
     * @return true the server with the given reference if blacklisted
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    public boolean isServerBlackListed(final String serverName)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        return ((ServerClusterViewManager) clusterViewManager).isServerBlackListed(new ServerRef(serverName));
    }

    /**
     * Adds the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public void addPooltoEmpty(final String objectName) throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        ((ServerClusterViewManager) clusterViewManager).addPoolToEmpty(objectName);
    }

    /**
     * Adds a server to the blacklist.
     * @param serverName a reference on a server
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    public void addServerToBlackList(final String serverName)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        ((ServerClusterViewManager) clusterViewManager).addServerToBlackList(new ServerRef(serverName));
    }

    /**
     * Removes the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public void removePoolToEmpty(final String objectName)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        ((ServerClusterViewManager) clusterViewManager).removePoolToEmpty(objectName);
    }

    /**
     * Removes a server from the blacklist.
     * @param serverName a reference on a server
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    public void removeServerFromBlackList(final String serverName)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        ((ServerClusterViewManager) clusterViewManager).removeServerFromBlackList(new ServerRef(serverName));
    }

    /**
     * Sets the maximal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName  a name of object
     * @param maxSize the maximal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public void setMaxPoolSize(final String objectName, final Integer maxSize)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        try {
            ((ServerClusterViewManager) clusterViewManager).setMaxPoolSize(objectName, maxSize);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Sets the minimal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName  a name of object
     * @param minSize the minimal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    public void setMinPoolSize(final String objectName, final Integer minSize)
    throws IllegalArgumentException, UnsupportedOperationException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        try {
            ((ServerClusterViewManager) clusterViewManager).setMinPoolSize(objectName, minSize);
        } catch (ObjectNotFoundException e) {
            LOGGER.error("Object not found: {0}",objectName, e);
            throw new IllegalArgumentException("Object not found: "+objectName+"\n"+e.getMessage());
        }
    }

    /**
     * Returns the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @return the load-factor for the server with the given address
     * @throws IllegalArgumentException if none server has the given address
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    public Integer getLoadFactor(final String serverRef)
    throws IllegalArgumentException, MalformedURLException, UnknownHostException {
        try {
            return clusterViewManager.getLoadFactor(new ServerRef(serverRef));
        } catch(ServerNotFoundException e) {
            LOGGER.error("Server not found: {0}",serverRef, e);
            throw new IllegalArgumentException("Server not found: "+serverRef+"\n"+e.getMessage());
        }
    }

    /**
     * Sets the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @param loadFactor the load-factor for the server with the given address
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    public void setLoadFactor(final String serverRef, final Integer loadFactor)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        ((ServerClusterViewManager) clusterViewManager).setLoadFactor(new ServerRef(serverRef), loadFactor);
    }

    /**
     * @return the numbers of clients connected to a provider of the cluster view
     */
    public Integer getNbClientsConnectedToProvider() {
        if(!(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Clients cannot call this method");
            throw new UnsupportedOperationException("Clients cannot call this method");
        }
        return ((ServerClusterViewManager) clusterViewManager).getNbClientsConnectedToProvider();
    }

}
