/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: CMIAdminConnectorManager.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.admin;

import java.io.IOException;

import java.net.MalformedURLException;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXConnectorServerFactory;
import javax.management.remote.JMXServiceURL;

import net.jcip.annotations.NotThreadSafe;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This helper class allow to start a JMX remote connector allowing to connect
 * remote applications.
 * @author new CMI Team
 */
@NotThreadSafe
public final class CMIAdminConnectorManager {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(CMIAdminConnectorManager.class);

    /**
     * Map protocol -> RMI connector.
     */
    private static Map<String, JMXConnectorServer> jmxConnectorServers =
        new HashMap<String, JMXConnectorServer>();

    /**
     * ObjectName for the connector.
     */
    private static final String DEFAULT_NAME_CONNECTOR = "connectors:name=CMI_JMXRemoteConnector";

    /**
     * Prefix to construct a service URL.
     */
    private static final String PRE_PROVIDER_URL = "service:jmx:";

    /**
     * Name of the JMX connector server.
     */
    private static final String SERVER_NAME = "/cmi_connector";


    /**
     * Hashtable mapping between protocol name and scheme name.
     */
    private static final Properties MAPPING = new Properties();

    static {
        // Initialize the hashtable with the supported protocols
        MAPPING.setProperty("jrmp", "rmi");
        MAPPING.setProperty("iiop", "iiop");
        MAPPING.setProperty("irmi", "rmi");
    }

    /**
     * Constructor.
     */
    private CMIAdminConnectorManager() {}

    /**
     * Build a new JMX Remote connector.
     * @param protocol a protocol name
     * @param url JMX remote URL
     * @param mbeanserver the MBean server
     * @throws CMIMBeanConfigException if JMX connector can't be built.
     */
    private static void initURL(final String protocol, final String url, final MBeanServer mbeanserver)
            throws CMIMBeanConfigException {

        // Create connector
        Map<String, String> environment = new HashMap<String, String>();
        JMXServiceURL jmxServiceURL = null;
        try {
            jmxServiceURL = new JMXServiceURL(url);
        } catch (MalformedURLException e) {
            logger.error("can't create a jmx service url with url: {0}", url, e);
            throw new CMIMBeanConfigException(
                    "Cannot create jmxservice url with url '" + url + "'.", e);
        }
        environment.put("jmx.remote.jndi.rebind", "true");
        try {
            jmxConnectorServers.put(protocol,
                JMXConnectorServerFactory.newJMXConnectorServer(jmxServiceURL, environment, mbeanserver));
        } catch (IOException e) {
            logger.error("Cannot create new JMX Connector", e);
            throw new CMIMBeanConfigException(
                    "Cannot create new JMX Connector", e);
        }
    }

    /**
     * Start a JMX connector (used to do remote administration).
     * @param protocol a protocol name
     * @param providerURL a provider URL to register the connector
     * @throws CMIMBeanConfigException if the connector can't be started.
     */
    public static void startConnector(final String protocol, final String providerURL) throws CMIMBeanConfigException {

        String url =
            PRE_PROVIDER_URL+MAPPING.getProperty(protocol)+":///jndi/"+providerURL+SERVER_NAME;

        MBeanServer mbeanServer = MBeanUtils.getMBeanServer();

        // Create connector if null
        if (jmxConnectorServers.get(protocol) == null) {
            if (url == null) {
                logger.error("The url cannot be empty");
                throw new CMIMBeanConfigException("The url cannot be empty");
            } else {
                initURL(protocol, url, mbeanServer);
            }
        }
        // Create MBean for this connector
        ObjectName connectorServerName = null;
        String objName = null;
        try {
            objName = DEFAULT_NAME_CONNECTOR;
            connectorServerName = new ObjectName(objName);
        } catch (Exception e) {
            logger.error("Cannot create ObjectName with name: {0}", objName, e);
            throw new CMIMBeanConfigException(
                    "Cannot create ObjectName with name '" + objName + "'", e);
        }

        // register it
        try {
            mbeanServer.registerMBean(jmxConnectorServers.get(protocol), connectorServerName);
        } catch (Exception e) {
            logger.error("Cannot register Mbean with the name: {0}",
                    connectorServerName, e);
            throw new CMIMBeanConfigException(
                    "Cannot register Mbean with the name '"
                            + connectorServerName + "'", e);
        }

        // Start connector
        try {
            jmxConnectorServers.get(protocol).start();
            logger.info("JMXServerConnector is started on {0}.", url);
        } catch (IOException e) {
            logger.error("Cannot start the jmx connector", e);
            throw new CMIMBeanConfigException(
                    "Cannot start the jmx connector", e);
        }
    }

    /**
     * @param protocolName a name of protocol
     * @return the JMX service URL for the given protocol or null if the protocol doesn't exist
     */
    public static JMXServiceURL getJMXServiceURL(final String protocolName) {
        JMXConnectorServer jmxConnectorServer = jmxConnectorServers.get(protocolName);
        if(jmxConnectorServers == null) {
            return null;
        } else {
            return jmxConnectorServer.getAddress();
        }
    }
}
