/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MBeanUtils.java 1521 2007-12-03 13:06:51Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.carol.cmi.admin;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.config.CMIConfig;
import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


/**
 * This class is used to manage MBean registration.
 * @author The new CMI team
 */
@ThreadSafe
public final class MBeanUtils {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(MBeanUtils.class);

    /**
     * MBeanServer.
     */
    private static volatile MBeanServer mbeanServer;

    /**
     * MBean domain name.
     */
    private static volatile String mbeanDomainName;

    /**
     * The agent identifier of the MBeanServer to retrieve.
     */
    private static volatile String agentId;

    /**
     * MBean server name.
     */
    private static String mbeanServerName;

    /**
     * True if it is already initialized.
     */
    private static boolean initialized = false;

    /**
     * True if CMIMBean is already registered.
     */
    private static boolean registered = false;

    /**
     * Contains the additional MBeans.
     */
    private static final Set<String> XTRAMBEANS = new HashSet<String>();

    /**
     * ObjectName for the CMI MBean.
     */
    private static ObjectName cmiAdminObjectName = null;

    /**
     * Utility class, so no constructor.
     */
    private MBeanUtils() {}

    /**
     * Initialize the MBeanServer.
     * Use an already existing server when CMI is embedded, else create a new MBean server.
     * @throws CMIMBeanConfigException if CMI is embedded but no MBean server is available
     */
    private static void initMBeanServer() throws CMIMBeanConfigException {

        // Gets the first available MBean Server
        List<?> mbeanServers = MBeanServerFactory.findMBeanServer(agentId);
        if (mbeanServers.size() == 0) {
            if(CMIConfig.isEmbedded()) {
                logger.error("CMI is embedded but no MBeanServer is available !");
                throw new CMIMBeanConfigException("CMI is embedded but no MBeanServer is available !");
            }
            logger.debug("Creating a new MBeanServer...");
            // Needs to create an MBean server
            mbeanServer = MBeanServerFactory.createMBeanServer(mbeanDomainName);
        } else {
            if (mbeanServers.size() > 1) {
                logger.info("Many MBeanServer available: taking the first of the list");
            }
            mbeanServer = (MBeanServer) mbeanServers.get(0);
        }
    }

    /**
     * Set the MBean parameters when CMI is embedded.
     * @param domainName the name of the JOnAS domain
     * @param serverName the name of the server for creating MBeans
     * @param agentId the agent identifier of the MBeanServer to retrieve
     */
    public static synchronized void setMBeanParameters(final String domainName, final String serverName, final String agentId) {
        mbeanDomainName = domainName;
        mbeanServerName = serverName;
        MBeanUtils.agentId = agentId;
        CMIConfig.setEmbedded();
    }

    /**
     * Get the MBean server.
     * @return MBean server
     */
    public static MBeanServer getMBeanServer() {
        return mbeanServer;
    }

    /**
     * Build a JMX Object Name.
     * @param name a value for the attribute name
     * @return a JMX Object Name
     */
    private static String buildObjectName(final String name) {
        StringBuffer sb = new StringBuffer(mbeanDomainName);
        sb.append(":type=cmi");
        sb.append(",name=" + name);
        if(CMIConfig.isEmbedded()) {
            sb.append(",J2EEServer=" + mbeanServerName);
        }
        return sb.toString();
    }

    /**
     * Get the MBean domain name.
     * @return the MBean domain name
     */
    public static String getMBeanDomainName() {
        return mbeanDomainName;
    }

    /**
     * Register the CMIMBean.
     * @param clusterViewManager the manager of the cluster view to use
     * @throws CMIMBeanConfigException if CMIAdmin MBean cannot be registered
     */
    public static synchronized void registerCMIMBean(final ClusterViewManager clusterViewManager)
    throws CMIMBeanConfigException {

        if(!initialized) {
            logger.error("CMIMBean is not yet initialized: cannot register it");
            throw new CMIMBeanConfigException("CMIMBean is not yet initialized: cannot register it");
        }
        if(registered) {
            logger.error("CMIMBean is already registered");
            throw new CMIMBeanConfigException("CMIMBean is already registered");
        }

        // Creates an ObjectName to register CMIAdmin

        String cmiAdminObjectNameRepresentation = buildObjectName(CMIConfig.getCMIAdminMBeanName());
        try {
            cmiAdminObjectName = new ObjectName(cmiAdminObjectNameRepresentation);
        } catch (MalformedObjectNameException e) {
            logger.error("Cannot build an ObjectName for CMIAdmin with representation {0}",
                    cmiAdminObjectNameRepresentation, e);
            throw new CMIMBeanConfigException("Cannot build an ObjectName for CMIAdmin with representation "
                    +cmiAdminObjectNameRepresentation, e);
        }

        // Creates the CMIAdminMBean
        CMIAdminMBean cmiAdmin = CMIAdmin.getCMIAdmin(cmiAdminObjectName, clusterViewManager);

        // Registers it with its ObjectName
        try {
            mbeanServer.registerMBean(cmiAdmin, cmiAdminObjectName);
        } catch (Exception e) {
            logger.error("Cannot register the CMIAdmin MBean", e);
            throw new CMIMBeanConfigException("Cannot register the CMIAdmin MBean", e);
        }
        registered = true;
    }


    /**
     * Unregister the CMIMBean.
     * @throws CMIMBeanConfigException if CMIAdmin MBean cannot be unregistered
     */
    public static void unregisterCMIMBean() throws CMIMBeanConfigException {
        if(registered) {
            try {
                mbeanServer.unregisterMBean(cmiAdminObjectName);
            } catch (Exception e) {
                logger.error("Cannot unregister the CMIAdmin MBean", e);
                throw new CMIMBeanConfigException("Cannot unregister the CMIAdmin MBean", e);
            }
            CMIAdmin.setCMIAdmin(null);
            registered = false;
        }
    }

    /**
     * Registers an additional MBean.
     * @param name its name
     * @param mbean the object
     * @throws CMIMBeanConfigException if the MBean cannot be registered
     */
    public static synchronized ObjectName registerXtraMBean(final String name, final Object mbean)
    throws CMIMBeanConfigException {
        if(!registered) {
            logger.error("CMIMBean is not already registered");
            throw new CMIMBeanConfigException("CMIMBean is not already registered");
        }
        if(XTRAMBEANS.contains(name)) {
            logger.error("{0} is already registered", name);
            throw new CMIMBeanConfigException(name+" is already registered");
        }
        // Creates an ObjectName to register the MBean
        ObjectName objectName = null;
        String objectNameRepresentation = buildObjectName(name);
        try {
            objectName = new ObjectName(objectNameRepresentation);
        } catch (MalformedObjectNameException e) {
            logger.error("Cannot build an ObjectName with representation {0}",
                    objectNameRepresentation, e);
            throw new CMIMBeanConfigException("Cannot build an ObjectName with representation "
                    +objectNameRepresentation, e);
        }
        // Registers it with its ObjectName
        try {
            mbeanServer.registerMBean(mbean, objectName);
        } catch (Exception e) {
            logger.error("Cannot register the MBean", e);
            throw new CMIMBeanConfigException("Cannot register the MBean", e);
        }
        XTRAMBEANS.add(name);
        return objectName;
    }

    /**
     * Initializes CMIMBean.
     * @throws CMIMBeanConfigException if a MBean server cannot be retrieved
     */
    public static synchronized void initCMIMBean() throws CMIMBeanConfigException {

        // Checks if it is not already initialized
        if(initialized) {return; }

        if(!CMIConfig.isEmbedded()) {
            // Standalone mode: reads MBeanDomainName from cmi.properties
            mbeanDomainName = CMIConfig.getMBeanDomainName();
            // No MBeanServerName
            mbeanServerName = null;
        }
        logger.debug("MBeanDomainName: {0}", mbeanDomainName);
        logger.debug("MBeanServerName: {0}", mbeanServerName);

        // Get a MBean server
        initMBeanServer();
        initialized = true;
    }

}
