/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: ClusterViewManager.java 1332 2007-10-17 12:45:37Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.controller.common;

import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.ejb.EJBObject;

import org.ow2.carol.cmi.ha.SessionId;
import org.ow2.carol.cmi.lb.policy.ILBPolicy;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.CMIReferenceable;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;
import org.ow2.util.pool.api.Pool;

/**
 * Implementations of this interface define a manager of the cluster view.
 * A CMIContext delegates it the management of the cluster view.
 * @author The new CMI team
 * @see CMIContext
 */
public interface ClusterViewManager {

    /**
     * Returns the name of class of policy for the object with the given name.
     * @param objectName a name of object
     * @return the name of class of policy for the object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getLBPolicyClassName(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the class that defines the policy for an object with the given name.
     * @param objectName a name of object
     * @return the class that defines the policy for an object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    Class<? extends ILBPolicy> getLBPolicyClass(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the name of class of strategy for the object with the given name.
     * @param objectName a name of object
     * @return the name of class of strategy for the object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getLBStrategyClassName(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the class that defines the strategy for an object with the given name.
     * @param objectName a name of object
     * @return the class that defines the strategy for an object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    Class<? extends ILBStrategy> getLBStrategyClass(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the date of properties for an object with the given name.
     * @param objectName a name of object
     * @return the date of properties for an object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    long getDateOfProperties(String objectName) throws ObjectNotFoundException;

    /**
     * Returns a value property of the LB policy for an object with the given name.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @return a value property of the LB policy for an object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    Object getPropertyForLBPolicy(String objectName, String propertyName) throws ObjectNotFoundException;

    /**
     * Returns the properties of the LB policy for an object with the given name.
     * @param objectName a name of object
     * @return the properties of the LB policy for an object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    Map<String, Object> getPropertiesForLBPolicy(String objectName) throws ObjectNotFoundException;

    /**
     * Returns a list of CMIReference for an object with the given name and protocol.
     * @param objectName a name of object
     * @param protocolName a name of protocol
     * @return a list of CMIReference for an object with the given name and protocol
     * @throws ObjectNotFoundException if none object has the given name for the given protocol
     */
    List<CMIReference> getCMIReferences(String objectName, String protocolName)
    throws ObjectNotFoundException;

    /**
     * Returns the interface of an object bound with the given name.
     * @param objectName a name of object
     * @return the interface of an object bound with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    Class<?> getInterface(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the business interface of an object bound with the given name (for ejb2 only).
     * @param objectName a name of object
     * @return the business interface of an object bound with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    Class<? extends EJBObject> getRemoteClass(String objectName) throws ObjectNotFoundException;

    /**
     * Return true if the object with the given name is replicated for high-availability.
     * The module 'ha' is so required.
     * @param objectName a name of object
     * @return true if the object with the given name is replicated for high-availability
     * @throws ObjectNotFoundException if none object has the given name
     */
    boolean isReplicated(String objectName) throws ObjectNotFoundException;

    /**
     * Returns a name of class that implements the interface InitialContextFactory for the given name of protocol.
     * @param protocolName a name of protocol
     * @return a name of class that implements the interface InitialContextFactory for the given name of protocol
     */
    String getInitialContextFactoryName(String protocolName);

    /**
     * Returns an instance of LB policy for the object that have the given name.
     * @param objectName a name of object
     * @return an instance of LB policy for the object that have the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    ILBPolicy<CMIReference> getLBPolicy(String objectName) throws ObjectNotFoundException;

    /**
     * Returns true if the object with the specified name is clustered.
     * @param objectName a name of object
     * @return true if the object with the specified name is clustered
     */
    boolean isClustered(String objectName);

    /**
     * Returns a pool of CMIReferenceable for an object with the given name.
     * @param objectName a name of object
     * @return a pool of CMIReferenceable for an object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    Pool<CMIReferenceable, CMIReference> getPool(String objectName) throws ObjectNotFoundException;

    /**
     * Associates a name of object with a pool of CMIReferenceable.
     * @param objectName a name of object
     * @param pool a pool of CMIReferenceable
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    void setPool(String objectName, Pool<CMIReferenceable, CMIReference> pool) throws ObjectNotFoundException;

    /**
     * Returns the name of cluster for the object with the given name.
     * @param objectName a name of object
     * @return the name of cluster for a object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getClusterName(String objectName) throws ObjectNotFoundException;

    /**
     * Adds an object with the given name to a set of objects that need to have their state up-to-date.
     * @param objectName a name of object
     * @throws ObjectNotFoundException if none object has the given name
     */
    void addObjectToWatch(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the minimal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the minimal size of pool of CMIReferenceable for a object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    int getMinPoolSize(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the maximal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the maximal size of pool of CMIReferenceable for a object with the given name
     * @throws ObjectNotFoundException if none object has the given name
     */
    int getMaxPoolSize(String objectName) throws ObjectNotFoundException;

    /**
     * Returns true if the pool for object with the given name should be empty.
     * @param objectName a name of object
     * @return true if the pool for object with the given name should be empty
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    boolean isPoolToEmpty(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the time between each update of the cluster view by clients.
     * @return the time between each update of the cluster view by clients
     */
    int getDelayToRefresh();

    /**
     * Returns the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @return the load-factor for the server with the given address
     * @throws ServerNotFoundException if none server has the given address
     */
    int getLoadFactor(ServerRef serverRef) throws ServerNotFoundException;

    /**
     * @return the UUID for this manager
     */
    UUID getUUID();

    /**
     * @return a new identifier of session
     */
    SessionId getSessionId();

}
