/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:ClientClusterViewProvider.java 966 2007-06-05 07:54:33Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.controller.provider;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.ow2.carol.cmi.lb.policy.ILBPolicy;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;

/**
 * Interface of providers which offers cluster view to the clients.
 * @author The new CMI team
 * @see org.ow2.carol.cmi.client.ClientClusterViewManagerImpl
 */
public interface ClientClusterViewProvider extends Remote {

    /**
     * Returns the name of the class of policy for the object with the given name.
     * @param objectName a name of object
     * @return the name of the class of policy for the object with the given name
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getLBPolicyClassName(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Gets the class of a given LB policy.
     * @param objectName a name of object
     * @return The corresponding LB policy class
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    Class<? extends ILBPolicy> getLBPolicyClass(String objectName)
    throws RemoteException, ObjectNotFoundException, ClassNotFoundException;

    /**
     * Returns the name of the class of strategy for the object with the given name.
     * @param objectName a name of object
     * @return the name of the class of strategy for the object with the given name
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getLBStrategyClassName(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Gets a class of strategy for the object with the given name.
     * @param objectName a name of object
     * @return a class of strategy for the object with the given name
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    Class<? extends ILBStrategy> getLBStrategyClass(String objectName)
    throws RemoteException, ObjectNotFoundException, ClassNotFoundException;

    /**
     * Returns the date of the last properties for a given object.
     * @param objectName a name of object
     * @return date of the last properties for a given object
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    long getDateOfProperties(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Returns the properties for the object with the given name.
     * @param objectName a name of object
     * @return the properties for the object with the given name
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    Map<String, Object> getPropertiesForLBPolicy(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Gets the nodes list of a given object and protocol.
     * @param objectName The name of object
     * @param protocolName The protocol of the client
     * @return The nodes list of the given object
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    List<CMIReference> getCMIReferences(String objectName, String protocolName)
    throws RemoteException, ObjectNotFoundException;

    /**
     * Returns the bytecode of the class that has the given binary name.
     * @param binaryName a binary name of a class
     * @return the bytecode of the class that has the given binary name
     * @throws RemoteException if the connection failed
     * @throws ClientClusterViewProviderException if an I/O exception happens on server over the reading of the bytecode
     */
    byte[] getBytecode(String binaryName) throws RemoteException, ClientClusterViewProviderException;

    /**
     * Gets the interface name for a given object.
     * @param objectName The name of object
     * @return the interface name of the given object
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getItfName(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Gets the business interface name for a given object.
     * @param objectName The name of object
     * @return the business interface name of the given object
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getBusinessName(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Return  the name of cluster that contains the specified object.
     * @param objectName a name of object
     * @return the name of cluster that contains the specified object
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    String getClusterName(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Returns true if the specified object is clustered.
     * @param name a name of object
     * @return true if the specified object is clustered
     * @throws RemoteException if there are errors on the protocol
     */
    boolean isClustered(String name) throws RemoteException;

    /**
     * Returns the delay to refresh the cluster view.
     * @return the delay to refresh the cluster view
     * @throws RemoteException if there are errors on the protocol
     */
    int getDelayToRefresh() throws RemoteException;

    /**
     * Returns the minimal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the minimal size of pool of CMIReferenceable for a object with the given name
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    int getMinPoolSize(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Returns the maximal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the maximal size of pool of CMIReferenceable for a object with the given name
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    int getMaxPoolSize(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Returns true if the pool for object with the given name should be empty.
     * @param objectName a name of object
     * @return true if the pool for object with the given name should be empty
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    boolean isPoolToEmpty(String objectName) throws RemoteException, ObjectNotFoundException;

    /**
     * Returns the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @return the load-factor for the server with the given address
     * @throws RemoteException if there are errors on the protocol
     * @throws ServerNotFoundException if none server has the given address
     */
    int getLoadFactor(ServerRef serverRef) throws RemoteException, ServerNotFoundException;

    /**
     * Register a new client (for statistic purposes).
     * @param uuid the Universally Unique Identifier of the client
     * @throws RemoteException if there are errors on the protocol
     */
    void registerClient(UUID uuid) throws RemoteException;

    /**
     * Return true if the object with the given name is replicated for high-availability.
     * The module 'ha' is so required.
     * @param objectName a name of object
     * @return true if the object with the given name is replicated for high-availability
     * @throws RemoteException if there are errors on the protocol
     * @throws ObjectNotFoundException if none object has the given name
     */
    boolean isReplicated(final String objectName) throws RemoteException, ObjectNotFoundException;

}
