/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: ServerClusterViewManager.java 1340 2007-10-23 08:48:56Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.controller.server;

import java.net.InetAddress;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.management.remote.JMXServiceURL;
import javax.naming.Context;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.info.ClusteredObjectInfo;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;

import org.ow2.carol.cmi.smart.api.SmartConnector;

/**
 * Interface for a manager of the cluster view at server-side.
 * @author The new CMI team
 */
public interface ServerClusterViewManager extends ClusterViewManager, SmartConnector {

    /**
     * Sets a new policy for a given object.
     * @param objectName a name of object
     * @param lbPolicyClassName a name of class of LB policy
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    void setLBPolicyClassName(String objectName, String lbPolicyClassName) throws ObjectNotFoundException;

    /**
     * Sets a new policy for a given object.
     * @param objectName a name of object
     * @param lbStrategyClassName a name of class of LB strategy
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    void setLBStrategyClassName(String objectName, String lbStrategyClassName) throws ObjectNotFoundException;

    /**
     * Sets properties for a given object.
     * @param objectName a name of object
     * @param properties properties for the LB policy of the given object
     * @throws ObjectNotFoundException if the specified object doesn't exist
     */
    void setPropertiesForLBPolicy(String objectName, Map<String, Object> properties)
    throws ObjectNotFoundException;

    /**
     * Sets a property for a given object.
     * @param objectName a name of object
     * @param propertyName a name of property for the LB policy of the given object
     * @param propertyValue a value for the given name of property
     * @throws ObjectNotFoundException if the specified object doesn't exist
     */
    void setPropertyForLBPolicy(String objectName, String propertyName, Object propertyValue)
    throws ObjectNotFoundException;

    /**
     * Sets the algorithm of load-balancing for the object with the given name.
     * @param objectName a name of object
     * @param lbPolicyClassName a name of class of LB policy
     * @param lbStrategyClassName a name of class of LB strategy
     * @param properties a set of properties
     * @throws ObjectNotFoundException if none object has the given name
     */
    void setAlgorithmForLBPolicy(final String objectName,
            final String lbPolicyClassName, final String lbStrategyClassName,
            final Map<String, Object> properties) throws ObjectNotFoundException;

    /**
     * Returns a list of CMIReference for an object with the given name.
     * @param objectName a name of object
     * @return a list of CMIReference for an object with the given name
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    List<CMIReference> getCMIReferences(String objectName) throws ObjectNotFoundException;

    /**
     * Removes a CMIReference from the cluster view.
     * @param cmiReference a reference on a clustered object
     */
    void removeCMIReference(CMIReference cmiReference);

    /**
     * Returns the set of name of cluster.
     * @return a set of name of cluster
     */
    Set<String> getClusterNames();

    /**
     * @return the set of name of object
     */
    Set<String> getObjectNames();

    /**
     * Returns the set of name of object for a given name of cluster.
     * @param clusterName a name of cluster
     * @return the set of name of object for a given name of cluster
     */
    Set<String> getObjectNames(String clusterName);

    /**
     * Adds an instance of a clustered object.
     * @param clusteredObjectInfo informations on a clustered object
     * @param cmiReference reference on a clustered object
     * @throws ServerClusterViewManagerException if the object cannot be added to the cluster.
     */
    void addObjectInstance(ClusteredObjectInfo clusteredObjectInfo, CMIReference cmiReference)
    throws ServerClusterViewManagerException;

    /**
     * Returns a name of interface of this object.
     * @param objectName a name of object
     * @return a name of interface of this object
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    String getItfName(String objectName) throws ObjectNotFoundException;

    /**
     * Returns a name of business interface of this object (for ejb2 only).
     * @param objectName a name of object
     * @return a name of business interface of this object
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    String getBusinessName(String objectName) throws ObjectNotFoundException;

    /**
     * Returns the list of Provider URL.
     * @param protocolName a name of protocol
     * @return providers of the cluster view for the protocol with the given name
     * @throws ServerClusterViewManagerException if none provider exists for the protocol with the given name
     */
    List<String> getProviderURLs(String protocolName) throws ServerClusterViewManagerException;

    /**
     * Adds a new protocol. This method also binds a provider of the cluster view for clients
     * and a dummy context, into the given instance of CMIContext.
     * @param initialContextFactoryName a InitialContextFactory for this protocol
     * @param serverRef a reference onto the local registry
     * @param cmiContext a instance of CMIContext to bind a provider of the cluster view for clients
     * and a dummy context
     * @throws ServerClusterViewManagerException if the protocol cannot be added
     */
    void addProtocol(String initialContextFactoryName, ServerRef serverRef, Context cmiContext)
    throws ServerClusterViewManagerException;

    /**
     * Returns the address of the local registries.
     * @return the address of the local registries
     */
    InetAddress getInetAddress();

    /**
     * Sets the time between each update of the cluster view by clients.
     * @param delay the time between each update of the cluster view by clients
     */
    void setDelayToRefresh(int delay);

    /**
     * Sets the minimal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName  a name of object
     * @param minPoolSize the minimal size of pool of CMIReferenceable for a object with the given name
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    void setMinPoolSize(String objectName, int minPoolSize) throws ObjectNotFoundException;

    /**
     * Sets the maximal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName  a name of object
     * @param maxPoolSize the maximal size of pool of CMIReferenceable for a object with the given name
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    void setMaxPoolSize(String objectName, int maxPoolSize) throws ObjectNotFoundException;

    /**
     * Removes the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     */
    void removePoolToEmpty(String objectName);

    /**
     * Adds the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     */
    void addPoolToEmpty(String objectName);

    /**
     * Returns true the server with the given reference if blacklisted.
     * @param serverRef a reference on a server
     * @return true the server with the given reference if blacklisted
     */
    boolean isServerBlackListed(ServerRef serverRef);

    /**
     * Adds a server to the blacklist.
     * @param serverRef a reference on a server
     */
    void addServerToBlackList(ServerRef serverRef);

    /**
     * Removes a server to the blacklist.
     * @param serverRef a reference on a server
     */
    void removeServerFromBlackList(ServerRef serverRef);

    /**
     * Sets the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @param loadFactor the load-factor for the server with the given address
     */
    void setLoadFactor(ServerRef serverRef, int loadFactor);

    /**
     * @return the protocols registered in the manager
     */
    Set<String> getProtocols();

    /**
     * @param protocolName a name of protocol
     * @return the JMX service URL to access to this MBean or null if the protocol doesn't exist
     */
    JMXServiceURL getJMXServiceURL(String protocolName);

    /**
     * @param protocolName a name of protocol
     * @return the reference on the local registry for the given protocol
     */
    ServerRef getRefOnLocalRegistry(String protocolName);

    /**
     * @param protocolName a name of protocol
     * @return the set of references on server connected to this server
     * @throws ServerClusterViewManagerException if the given protocol name doesn't exist
     */
    Set<ServerRef> getServerRefs(String protocolName) throws ServerClusterViewManagerException;

    /**
     * @return the numbers of clients connected to a provider of the cluster view
     */
    int getNbClientsConnectedToProvider();

    /**
     * Register a new client.
     * @param uuid the Universally Unique Identifier of the client
     */
    void registerClient(UUID uuid);

    /**
     * @return true if the replication manager is started
     */
    boolean isReplicationManagerStarted();

    /**
     * Set if the replication manager is started.
     * @param replicationManagerStarted true if the replication manager is started
     */
    void setReplicationManagerStarted(boolean replicationManagerStarted);

}
