/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:ClusteredBeanInfo.java 914 2007-05-25 16:48:16Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.info;

import java.util.Map;

import net.jcip.annotations.NotThreadSafe;

/**
 * Contain informations on a clustered object.
 * @author The new CMI team
 * @see ClusteredObject
 * @see CMIInfoExtractor
 */
@NotThreadSafe
public final class ClusteredObjectInfo {

    /**
     * Name of the interface that the object implements.
     * With ejb2 this is the home interface.
     */
    private final String itfName;

    /**
     * Name of the business interface.
     * Only for ejb2 support.
     */
    private final String businessName;

    /**
     * Name of the cluster where is deployed this object.
     */
    private final String clusterName;

    /**
     * Type of LB policy to use to access at this object.
     */
    private final String policyType;

    /**
     * Type of LB strategy to use to access at this object.
     * Can be null if no strategy is used.
     */
    private final String strategyType;

    /**
     * Minimal size for the pool of CMIReferenceable.
     */
    private final int minPoolSize;

    /**
     * Maximal size for the pool of CMIReferenceable.
     */
    private final int maxPoolSize;

    /**
     * Properties of the policy.
     * Values are casted.
     * Null if the policy doesn't support properties.
     */
    private final Map<String, Object> properties;

    /**
     * True if this object is replicated for high-availability (the module 'ha' is required).
     */
    private final boolean replicated;

    /**
     * Constructs informations for a clustered object with the default strategy for the given LB policy.
     * @param itfName a name of interface that the object implements
     * @param businessName the name of the business interface (only for ejb2)
     * @param clusterName a name of cluster
     * @param minPoolSize a minimal size for the pool of CMIReferenceable
     * @param maxPoolSize a maximal size for the pool of CMIReferenceable
     * @param policyType a type of policy for load-balancing
     * @param strategyType a type of strategy for load-balancing
     * @param properties properties of the policy
     * @param replicated true if this object is replicated for high-availability (the module 'ha' is required)
     */
    public ClusteredObjectInfo(
            final String itfName, final String businessName,
            final String clusterName, final int minPoolSize, final int maxPoolSize,
            final String policyType, final String strategyType,
            final Map<String, Object> properties, final boolean replicated) {
        this.itfName = itfName;
        this.businessName = businessName;
        this.clusterName = clusterName;
        this.policyType = policyType;
        this.strategyType = strategyType;
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
        this.properties = properties;
        this.replicated = replicated;
    }


    /**
     * @return a name of the cluster where is deployed this object
     */
    public String getClusterName() {
        return clusterName;
    }

    /**
     * @return Name of the interface that the object implements.
     */
    public String getItfName() {
        return itfName;
    }

    /**
     * @return the maximal size for the pool of CMIReferenceable
     */
    public int getMaxPoolSize() {
        return maxPoolSize;
    }

    /**
     * @return the minimal size for the pool of CMIReferenceable
     */
    public int getMinPoolSize() {
        return minPoolSize;
    }

    /**
     * @return the lbProperties
     */
    public Map<String, Object> getProperties() {
        return properties;
    }

    /**
     * @return the lbPolicyType
     */
    public String getPolicyType() {
        return policyType;
    }

    /**
     * @return the lbStrategyType
     */
    public String getStrategyType() {
        return strategyType;
    }

    /**
     * @return the businessName
     */
    public String getBusinessName() {
        return businessName;
    }

    /**
     * Return true if this object is replicated for high-availability (the module 'ha' is required)
     */
    public boolean isReplicated() {
        return replicated;
    }

    @Override
    public String toString() {
        return "ClusteredObject[itfName:" + itfName
            + (businessName!=null?",buisnessName:" + businessName:"")
            + (clusterName!=null?",clusterName:" + clusterName:"")
            + ",isReplicated:" + replicated
            + (policyType!=null?",policyType:" + policyType:"")
            + (strategyType!=null?",strategyType:" + strategyType:"")
            + (properties!=null?",properties:" + properties:"")
            + "]";
    }

}
