/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: ProviderURLParser.java 1262 2007-09-07 15:31:19Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.reference.util;

import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.Map;

import net.jcip.annotations.NotThreadSafe;

/**
 * Parses a provider URL.
 * @author The new CMI team
 */
@NotThreadSafe
public final class ProviderURLParser {

    /**
     * Index if the '/' character when no host and no port are set in the url.
     */
    private static final int URL_HOSTPORT_INDEX = 2;

    /**
     * Hashtable mapping between protocols and schemes.
     */
    private static final Map<String, String> MAPPING = new HashMap<String, String>();

    static {
        MAPPING.put("jrmp", "rmi");
        MAPPING.put("irmi", "rmi");
        MAPPING.put("iiop", "iiop");
    }

    /**
     * A name of protocol.
     */
    private final String protocol;

    /**
     * A provider URL.
     */
    private final String providerURL;

    /**
     * A scheme.
     */
    private String scheme;

    /**
     * A name of host.
     */
    private String hostName;

    /**
     * A number of port.
     */
    private int port;

    /**
     * Constructs a new parser of provider URL.
     * Check if the given protocol is consistent
     * @param protocol a name of protocol
     * @param providerURL a provider URL
     */
    public ProviderURLParser(final String protocol, final String providerURL) {
        if(protocol == null) {
            throw new IllegalArgumentException("The protocol name is null");
        }
        this.protocol = protocol;
        this.providerURL = providerURL;
    }

    /**
     * Constructs a new parser of provider URL.
     * @param providerURL a provider URL
     */
    public ProviderURLParser(final String providerURL) {
        protocol = null;
        this.providerURL = providerURL;
    }

    /**
     * Parse scheme. We check that scheme is consistent with protocol.
     * @param protocol the protocol for this scheme
     * @param inputscheme scheme
     * @throws MalformedURLException if malformed scheme
     */
    private void parseScheme(final String protocol, final String inputscheme) throws MalformedURLException {
        if (inputscheme.length() == 0) {
            throw new MalformedURLException("Scheme cannot be empty");
        } else {
            if (inputscheme.length() > 1 && inputscheme.endsWith(":")) {
                // non-empty scheme must contain at least one character and end with :
                scheme =
                    inputscheme.substring(0, inputscheme.length() - 1);
            } else {
                throw new MalformedURLException("badly formed scheme");
            }
            if (!isConsistent(protocol, scheme)) {
                throw new MalformedURLException("Invalid scheme : " + scheme);
            }
        }
    }

    /**
     * Compare a scheme and a protocol to check if there are consistents.
     * @param protocol A protocol (ie: jrmp, irmi, iiop)
     * @param scheme A scheme of URL (ie: rmi, iiop)
     * @return true is the scheme and the protocol are consistents.
     * @throws MalformedURLException
     */
    private boolean isConsistent(final String protocol, final String scheme) throws MalformedURLException {

        String sch = MAPPING.get(protocol);
        if(sch == null) {
            throw new MalformedURLException("Invalid protocol: "+protocol);
        }
        return sch.equals(scheme);
    }

    /**
     * Parse host port.
     * @param inputhostport host port
     * @throws MalformedURLException if parameters are malformed
     */
    private void parseHostPort(final String inputhostport)
        throws MalformedURLException {
        String inputhost = "";
        String inputport = "";
        int m = inputhostport.indexOf(':');
        if (m == -1) { // no port
            inputhost = inputhostport;
        } else {
            if (m == 0) { // no host
                if (inputhostport.length() > 1) {
                    // non-empty port preceded by /
                    inputport = inputhostport.substring(1);
                } else { // empty port preceded by : ; URL consists of //:
                    throw new MalformedURLException("non-empty port expected after :");
                }
            } else { // non-empty host, maybe non-empty port
                inputhost = inputhostport.substring(0, m);
                if (inputhostport.length() > m + 1) { // port specified
                    inputport = inputhostport.substring(m + 1);
                } else { // empty port preceded by : ; URL consists of //host:/
                    throw new MalformedURLException("non-empty port expected after :");
                }
            }
        }
        if (!inputhost.equals("")) {
            hostName = inputhost;
        }
        if (!inputport.equals("")) {
            try {
                port = Integer.parseInt(inputport);
            } catch (NumberFormatException e) {
                throw new MalformedURLException("port must be a number");
            }
        }
    }

    /**
     * Parse the provider URL.
     * @throws MalformedURLException if the provider URL is malformed
     */
    public void parse() throws MalformedURLException {
        if (providerURL == null || providerURL.length() == 0) {
            throw new MalformedURLException("null or empty registry URL");
        }
        if (providerURL.indexOf("//") == -1) {
            throw new MalformedURLException(
                    "badly formed registry URL " + providerURL);
        }
        try {
            if(protocol != null) {
                parseScheme(protocol, providerURL.substring(0, providerURL.indexOf("//")));
            }
            parseHostPort(providerURL.substring(providerURL.indexOf("//")+URL_HOSTPORT_INDEX));
        } catch (Exception e) {
            throw new MalformedURLException(
                    "badly formed registry URL " + providerURL);
        }
    }

    /**
     * @return the hostName
     */
    public String getHostName() {
        return hostName;
    }

    /**
     * @return the port
     */
    public int getPort() {
        return port;
    }

    /**
     * @return the scheme
     */
    public String getScheme() {
        return scheme;
    }
}
