/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:CMIAdminMBean.java 1006 2007-06-18 06:55:23Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.admin;

import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;
import javax.management.remote.JMXServiceURL;


/**
 * Definition of a MBean to monitor managers of cluster view.
 * In the case of a manager at server-side, we can also configure it with this MBean.
 * @author The new CMI team
 */
public interface CMIAdminMBean {

    /**
     * Returns the ObjectName binded in the MBean Server.
     * @return the ObjectName binded in the MBean Server
     */
    ObjectName getObjectName();

    /**
     * @param protocolName a name of protocol
     * @return the JMX service URL to access to this MBean
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    JMXServiceURL getJMXServiceURL(String protocolName) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * @return the protocols registered in the manager
     */
    Set<String> getProtocols();

    /**
     * @param protocolName a name of protocol
     * @return the reference on the local registry for the given protocol
     * @throws IllegalArgumentException if the given protocol name doesn't exist
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    String getRefOnLocalRegistry(String protocolName) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Returns a name of interface of this object.
     * @param objectName a name of object
     * @return a name of interface of this object
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    String getItfName(String objectName) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Returns the business interface name of an object bound with the given name (for ejb2 only).
     * If the object is not an ejb2, null is returned.
     * @param objectName a name of object
     * @return the business interface name of an object bound with the given name
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    String getBusinessName(String objectName) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * @param objectName a name of object
     * @return true if the object with the given name is replicated
     * @throws IllegalArgumentException if none object has the given name
     */
    boolean isReplicated(String objectName) throws IllegalArgumentException;

    /**
     * @param protocolName a name of protocol
     * @return the set of references on server connected to this server
     * @throws IllegalArgumentException if the given protocol name doesn't exist
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    Set<String> getServerRefsForProtocol(String protocolName) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * @return the set of clustered object names
     */
    Set<String> getObjectNames();

    /**
     * Returns the name of class of policy for the object with the given name.
     * @param objectName name of the object
     * @return the name of class of policy for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    String getLBPolicyClassName(String objectName) throws IllegalArgumentException;

    /**
     * Sets a new policy for a given object.
     * @param objectName a name of object
     * @param lbPolicyClassName a name of class of LB policy
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws ClassNotFoundException if the class is missing
     */
    void setLBPolicyClassName(String objectName, String lbPolicyClassName)
    throws IllegalArgumentException, UnsupportedOperationException, ClassNotFoundException;

    /**
     * Returns the name of class of strategy for the object with the given name.
     * @param objectName name of the object
     * @return the name of class of strategy for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    String getLBStrategyClassName(String objectName) throws IllegalArgumentException;

    /**
     * Sets a new strategy for a given object.
     * @param objectName a name of object
     * @param lbStrategyClassName a name of class of LB strategy
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws ClassNotFoundException if the class is missing
     */
    void setLBStrategyClassName(String objectName, String lbStrategyClassName)
    throws IllegalArgumentException, UnsupportedOperationException, ClassNotFoundException;

    /**
     * Returns the set of property names for the object with the given name.
     * @param objectName a name of object
     * @return the set of property names for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    Set<String> getPropertiesNamesForLBPolicy(String objectName) throws IllegalArgumentException;

    /**
     * Returns the set of property names (for which value is a list) for the object with the given name.
     * @param objectName a name of object
     * @return the set of property names for the object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    Set<String> getListPropertiesNamesForLBPolicy(String objectName) throws IllegalArgumentException;

    /**
     * Returns the value of the property with the given name.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @return the value of the property with the given name, or null if there is not property for this name
     * @throws IllegalArgumentException if none object has the given name, or if the value is a list
     */
    String getPropertyForLBPolicy(String objectName, String propertyName)
    throws IllegalArgumentException;

    /**
     * Returns the list of value of the property with the given name.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @return the list of value of the property with the given name, or null if there is not property for this name
     * @throws IllegalArgumentException if none object has the given name, or if the value is not a list
     */
    List<String> getListPropertyForLBPolicy(String objectName, String propertyName)
    throws IllegalArgumentException;

    /**
     * Sets a property for a given object. A property is either a String or a list of String.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @param propertyValue a value for the given name of property
     * @throws IllegalArgumentException if none object has the given name, or if the property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void setPropertyForLBPolicy(String objectName, String propertyName, String propertyValue)
    throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Sets a property for a given object. A property is either a String or a list of String.
     * @param objectName a name of object
     * @param propertyName a name of property
     * @param propertyValues a list of value for the given name of property
     * @throws IllegalArgumentException if none object has the given name, or if the property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void setListPropertyForLBPolicy(String objectName, String propertyName, List<String> propertyValues)
    throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Sets the properties for a given object. A property is either a String or a list of String.
     * @param objectName a name of object
     * @param properties properties a set of properties
     * @throws IllegalArgumentException if none object has the given name, or if a property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void setPropertiesForLBPolicy(String objectName, Map<String, Object> properties)
    throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Sets the algorithm of load-balancing for the object with the given name.
     * @param objectName a name of object
     * @param lbPolicyClassName a name of class of LB policy
     * @param lbStrategyClassName a name of class of LB strategy
     * @param properties a set of properties
     * @throws IllegalArgumentException if none object has the given name, or if a property doesn't exist or has an invalid type
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws ClassNotFoundException if the class is missing
     */
    void setAlgorithmForLBPolicy(String objectName, String lbPolicyClassName,
            String lbStrategyClassName, Map<String, Object> properties)
    throws IllegalArgumentException, UnsupportedOperationException, ClassNotFoundException;

    /**
     * Returns a list of String representing a ServerRef for an object with the given name and protocol.
     * @param objectName a name of object
     * @param protocolName a name of protocol
     * @return a list of String representing a ServerRef for an object with the given name and protocol
     * @throws IllegalArgumentException if none object has the given name
     */
    List<String> getServerRefs(String objectName, String protocolName) throws IllegalArgumentException;

    /**
     * Returns a list of String representing a ServerRef for an object with the given name.
     * @param objectName a name of object
     * @return a list of String representing a ServerRef for an object with the given name
     * @throws IllegalArgumentException if none object has the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    List<String> getServerRefs(String objectName)
    throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Returns the set of name of cluster.
     * @return the set of name of cluster
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    Set<String> getClusterNames() throws UnsupportedOperationException;

    /**
     * Returns the set of object names included in the given cluster.
     * @param clusterName The cluster name
     * @return a set of object names included in the given cluster
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws IllegalArgumentException if none cluster has the given name
     */
    Set<String> getObjectNames(String clusterName) throws UnsupportedOperationException, IllegalArgumentException;

    /**
     * Returns the time between each update of the cluster view by clients.
     * @return the time between each update of the cluster view by clients
     */
    Integer getDelayToRefresh();

    /**
     * Sets the time between each update of the cluster view by clients.
     * @param delay the time between each update of the cluster view by clients
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void setDelayToRefresh(Integer delay) throws UnsupportedOperationException;

    /**
     * Returns the name of cluster for the object with the given name.
     * @param objectName a name of object
     * @return the name of cluster for a object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    String getClusterName(String objectName) throws IllegalArgumentException;

    /**
     * Returns the minimal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the minimal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    Integer getMinPoolSize(String objectName) throws IllegalArgumentException;

    /**
     * Returns the maximal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName a name of object
     * @return the maximal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if none object has the given name
     */
    Integer getMaxPoolSize(String objectName) throws IllegalArgumentException;

    /**
     * Sets the minimal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName  a name of object
     * @param minPoolSize the minimal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void setMinPoolSize(String objectName, Integer minPoolSize) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Sets the maximal size of pool of CMIReferenceable for a object with the given name.
     * @param objectName  a name of object
     * @param maxPoolSize the maximal size of pool of CMIReferenceable for a object with the given name
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void setMaxPoolSize(String objectName, Integer maxPoolSize) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Adds a server to the blacklist.
     * @param serverName a reference on a server
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    void addServerToBlackList(String serverName)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException;

    /**
     * Removes a server from the blacklist.
     * @param serverName a reference on a server
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    void removeServerFromBlackList(String serverName)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException;

    /**
     * Returns true the server with the given reference if blacklisted.
     * @param serverName a reference on a server
     * @return true the server with the given reference if blacklisted
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    boolean isServerBlackListed(String serverName)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException;

    /**
     * Adds the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void addPooltoEmpty(String objectName) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Removes the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     * @throws IllegalArgumentException if no object is bound with the given name
     * @throws UnsupportedOperationException if the used manager is at client-side
     */
    void removePoolToEmpty(String objectName) throws IllegalArgumentException, UnsupportedOperationException;

    /**
     * Returns true if the pool for object with the given name should be empty.
     * @param objectName a name of object
     * @return true if the pool for object with the given name should be empty
     * @throws IllegalArgumentException if no object is bound with the given name
     */
    boolean isPoolToEmpty(String objectName) throws IllegalArgumentException;

    /**
     * Returns the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @return the load-factor for the server with the given address
     * @throws IllegalArgumentException if none server has the given address
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    Integer getLoadFactor(String serverRef) throws IllegalArgumentException, MalformedURLException, UnknownHostException;

    /**
     * Sets the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @param loadFactor the load-factor for the server with the given address
     * @throws UnsupportedOperationException if the used manager is at client-side
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    void setLoadFactor(String serverRef, Integer loadFactor)
    throws UnsupportedOperationException, MalformedURLException, UnknownHostException;


    /**
     * @return the numbers of clients connected to a provider of the cluster view
     */
    Integer getNbClientsConnectedToProvider();

}
