/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:ILBPolicy.java 949 2007-06-02 17:24:33Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.policy;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.List;

import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.lb.NoLoadBalanceableException;
import org.ow2.carol.cmi.lb.decision.DecisionManager;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;

/**
* Interface of the policies for load-balancing.
* @param <T> The type of object that was load-balanced
* @author The new CMI team
*/
public interface ILBPolicy<T extends LoadBalanceable> extends Serializable {

    /**
     * Chooses a load-balanceable among the list of load-balanceables.
     * @param loadBalanceables a list of load-balanceables
     * @throws NoLoadBalanceableException if no server is available
     * @return the chosen load-balanceable
     */
    T choose(List<T> loadBalanceables) throws NoLoadBalanceableException;

    /**
     * Sets a strategy to modify the behavior of this policy.
     * @param lbStrategy a strategy of load-balancing
     */
    void setStrategy(ILBStrategy<T> lbStrategy);

    /**
     * Returns a decision when an exception is thrown during an access to a registry for a given load-balanceable.
     * @param loadBalanceable the load-balanceable that have caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an access to a registry for a given load-balanceable
     */
    DecisionManager<Void> onLookupException(T loadBalanceable, Throwable thr);

    /**
     * Returns a decision when an exception is thrown during an invocation for a given load-balanceable.
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param loadBalanceable the load-balanceable that have caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an invocation for a given load-balanceable
     */
    DecisionManager<Void> onInvokeException(Method method, Object[] parameters, T loadBalanceable, Throwable thr);

    /**
     * Returns a decision when the invocation of a remote method ends.
     * @param <ReturnType> the type of the returned value
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param loadBalanceable the load-balanceable used for the invocation
     * @param retVal the returned value
     * @return the decision when the invocation of a remote method ends
     */
    <ReturnType> DecisionManager<ReturnType> onReturn(Method method, Object[] parameters, T loadBalanceable, ReturnType retVal);
}
