/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ClusteredObjectInfo.java 1649 2008-02-28 17:36:13Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.info;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.ejb.EJBObject;

import org.ow2.carol.cmi.lb.policy.ILBPolicy;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.util.pool.api.IPoolConfiguration;

import net.jcip.annotations.NotThreadSafe;

/**
 * Contain informations on a clustered object.
 * @author The new CMI team
 * @see CMIInfoExtractor which creates instances of this
 */
@NotThreadSafe
public final class ClusteredObjectInfo {

    /**
     * Interface implemented by the object.
     * With ejb2 this is the home interface.
     */
    private final Class<?> itfClass;

    /**
     * Business interface.
     * Only for ejb2 support.
     */
    private final Class<? extends EJBObject> businessClass;

    /**
     * Name of the cluster where is deployed this object.
     */
    private final String clusterName;

    /**
     * Type of LB policy to use to access at this object.
     */
    @SuppressWarnings("unchecked")
    private final Class<? extends ILBPolicy> policyType;

    /**
     * Type of LB strategy to use to access at this object.
     * Can be null if no strategy is used.
     */
    @SuppressWarnings("unchecked")
    private final Class<? extends ILBStrategy> strategyType;

    /**
     * Configuration for the pool of CMIReferenceable.
     */
    private final IPoolConfiguration poolConfiguration;

    /**
     * Properties of the policy.
     * Values are casted.
     * Null if the policy doesn't support properties.
     */
    private final Map<String, Object> properties;

    /**
     * True if this object has a state.
     */
    private final boolean stateful;

    /**
     * True if the state of this object is replicated for high-availability.
     */
    private final boolean replicated;

    /**
     * Classnames of the application exceptions.
     */
    private final Set<String> applicationExceptionNames;

    /**
     * Constructs informations for a clustered object with the default strategy for the given LB policy.
     * @param itfClass interface implemented by the object
     * @param businessClass Business interface (only for ejb2)
     * @param clusterName a name of cluster
     * @param poolConfiguration configuration for the pool of CMIReferenceable
     * @param policyType a type of policy for load-balancing
     * @param strategyType a type of strategy for load-balancing
     * @param properties properties of the policy
     * @param stateful true if this object has a state
     * @param replicated true if the state of this object is replicated for high-availability
     * @param applicationExceptionNames classnames of the application exceptions
     */
    @SuppressWarnings("unchecked")
    public ClusteredObjectInfo(
            final Class<?> itfClass,
            final Class<? extends EJBObject> businessClass,
            final String clusterName,
            final IPoolConfiguration poolConfiguration,
            final Class<? extends ILBPolicy> policyType,
            final Class<? extends ILBStrategy> strategyType,
            final Map<String, Object> properties,
            final boolean stateful,
            final boolean replicated,
            final Set<String> applicationExceptionNames) {
        this.itfClass = itfClass;
        this.businessClass = businessClass;
        this.clusterName = clusterName;
        this.policyType = policyType;
        this.strategyType = strategyType;
        this.poolConfiguration = poolConfiguration;
        this.properties = properties;
        this.stateful = stateful;
        this.replicated = replicated;
        if(applicationExceptionNames == null) {
            this.applicationExceptionNames = new HashSet<String>();
        } else {
            this.applicationExceptionNames = new HashSet<String>(applicationExceptionNames);
        }
    }


    /**
     * @return a name of the cluster where is deployed this object
     */
    public String getClusterName() {
        return clusterName;
    }

    /**
     * @return Name of the interface that the object implements.
     */
    public Class<?> getItfClass() {
        return itfClass;
    }

    /**
     * @return the lbPolicyType
     */
    @SuppressWarnings("unchecked")
    public Class<? extends ILBPolicy> getPolicyType() {
        return policyType;
    }

    /**
     * @return the lbStrategyType
     */
    @SuppressWarnings("unchecked")
    public Class<? extends ILBStrategy> getStrategyType() {
        return strategyType;
    }

    /**
     * @return the businessName
     */
    public Class<? extends EJBObject> getBusinessClass() {
        return businessClass;
    }

    /**
     * @return the lbProperties
     */
    public Map<String, Object> getProperties() {
        return properties;
    }

    /**
     * @return the configuration for the pool of CMIReferenceable
     */
    public IPoolConfiguration getPoolConfiguration() {
        return poolConfiguration;
    }

    /**
     * Return true if this object has a state
     */
    public boolean hasState() {
        return stateful;
    }

    /**
     * @return true if the state of this object is replicated for high-availability
     */
    public boolean isReplicated() {
        return replicated;
    }

    /**
     * @return classnames of the application exceptions
     */
    public Set<String> getApplicationExceptionNames() {
        return applicationExceptionNames;
    }

    @Override
    public boolean equals(final Object object) {
        if(object == null || !(object instanceof ClusteredObjectInfo)) {
            return false;
        }
        ClusteredObjectInfo clusteredObjectInfo = (ClusteredObjectInfo) object;
        return (itfClass == null ? clusteredObjectInfo.itfClass == null
                    : itfClass.equals(clusteredObjectInfo.itfClass))
                && (businessClass == null ? clusteredObjectInfo.businessClass == null
                    : businessClass.equals(clusteredObjectInfo.businessClass))
                && (clusterName == null ? clusteredObjectInfo.clusterName == null :
                    clusterName.equals(clusteredObjectInfo.clusterName))
                && (poolConfiguration == null ? clusteredObjectInfo.poolConfiguration == null
                    : poolConfiguration.equals(clusteredObjectInfo.poolConfiguration))
                && policyType.equals(clusteredObjectInfo.policyType)
                && (strategyType == null ? clusteredObjectInfo.strategyType == null
                    : strategyType.equals(clusteredObjectInfo.strategyType))
                && (properties == null ? clusteredObjectInfo.properties == null
                    : properties.equals(clusteredObjectInfo.properties))
                && stateful == clusteredObjectInfo.stateful
                && replicated == clusteredObjectInfo.replicated
                && (applicationExceptionNames == null ? clusteredObjectInfo.applicationExceptionNames == null :
                        applicationExceptionNames.equals(clusteredObjectInfo.applicationExceptionNames));
    }

    public int hashCode() {
        return (itfClass == null ? 0 : itfClass.getName().hashCode())
         + (businessClass == null ? 0 : businessClass.getName().hashCode())
         + (clusterName == null ? 0 : clusterName.hashCode())
         + (poolConfiguration == null ? 0 : poolConfiguration.hashCode())
         + (policyType == null ? 0 : policyType.hashCode())
         + (strategyType == null ? 0 : strategyType.hashCode())
         + (properties == null ? 0 : properties.hashCode())
         + (stateful ? 1 : 0)
         + (replicated ? 1 : 0)
         + (applicationExceptionNames == null ? 0 : applicationExceptionNames.hashCode());
    }

    @Override
    public String toString() {
        return "ClusteredObject[itfClass:" + itfClass
            + ",businessClass:" + businessClass
            + ",clusterName:" + clusterName
            + ",poolConfiguration: " + poolConfiguration
            + ",policyType:" + policyType
            + ",strategyType:" + strategyType
            + ",properties:" + properties
            + ",stateful:" + stateful
            + ",replicated:" + replicated
            + ",applicationExceptionNames:" + applicationExceptionNames
            + "]";
    }

}
