/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: CMIReference.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.reference;

import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.UnknownHostException;

import net.jcip.annotations.Immutable;

import org.ow2.carol.cmi.lb.LoadBalanceable;

/**
 * Represents a reference on a clustered object that is deployed on a server.
 * @author The new CMI team
 */
@Immutable
public final class CMIReference implements Serializable, LoadBalanceable {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = -2546573353023565090L;

    /**
     * Server on which is deployed the referenced object.
     */
    private final ServerRef serverRef;

    /**
     * Name of the clustered object that is referenced.
     */
    private final String objectName;

    /**
     * Constructs an instance of CMIReference.
     * @param protocol a name of protocol to access at the clustered object
     * @param providerURL a provider URL to access at the clustered object
     * @param objectName a name of object
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the provided address can not be determined
     */
    public CMIReference(final String protocol, final String providerURL, final String objectName)
    throws MalformedURLException, UnknownHostException {
        serverRef = new ServerRef(protocol, providerURL);
        this.objectName = objectName;
    }

    /**
     * Constructs an instance of CMIReference.
     * @param serverRef a reference on a server
     * @param objectName a name of object
     */
    public CMIReference(final ServerRef serverRef, final String objectName) {
        this.serverRef = serverRef;
        this.objectName = objectName;
    }

    /**
     * @return Name of the clustered object
     */
    public String getObjectName() {
        return objectName;
    }

    @Override
    public String toString() {
        return serverRef.getProviderURL()+"/"+objectName;
    }

    @Override
    public boolean equals(final Object obj) {
        if(obj == null || !(obj instanceof CMIReference)) {
            return false;
        }
        CMIReference cmiReference = (CMIReference) obj;
        return objectName.equals(cmiReference.objectName)
                && serverRef.equals(cmiReference.serverRef);
    }

    @Override
    public int hashCode() {
        return serverRef.hashCode()+objectName.hashCode();
    }

    /**
     * @return a reference on server on which is deployed the referenced object
     */
    public ServerRef getServerRef() {
        return serverRef;
    }

}
