/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: ServerRef.java 1604 2008-01-28 22:14:00Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.reference;

import java.io.Serializable;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.UnknownHostException;

import net.jcip.annotations.Immutable;

import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.reference.util.ProviderURLParser;


/**
 * Represents a reference on a server.
 * @author The new CMI team
 */
@Immutable
public class ServerRef implements Serializable, LoadBalanceable {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = -1732709011433671653L;

    /**
     * InetAdress of the server.
     */
    private final InetAddress inetAddress;

    /**
     * Port of the provider URL.
     */
    private final int port;

    /**
     * Provider URL.
     */
    private final String providerURL;

    /**
     * Protocol to access at the clustered object.
     */
    private final String protocol;


    /**
     * Construct a new reference for a server.
     * Check if the given protocol is consistent with the provider URL.
     * @param protocol a name of protocol
     * @param providerURL a provider URL
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    public ServerRef(final String protocol, final String providerURL) throws MalformedURLException, UnknownHostException {
        this.protocol = protocol;
        ProviderURLParser providerURLParser = new ProviderURLParser(protocol, providerURL);
        providerURLParser.parse();
        inetAddress = InetAddress.getByName(providerURLParser.getHostName());
        port = providerURLParser.getPort();
        this.providerURL =
            providerURLParser.getScheme() + "://"
            + inetAddress.getHostAddress() + ":" + port;
    }

    /**
     * Construct a new reference for a server.
     * @param providerURL a provider URL
     * @throws MalformedURLException if the URL is malformed
     * @throws UnknownHostException if the given host cannot be resolved
     */
    public ServerRef(final String providerURL) throws MalformedURLException, UnknownHostException {
        protocol = null;
        ProviderURLParser providerURLParser = new ProviderURLParser(providerURL);
        providerURLParser.parse();
        inetAddress = InetAddress.getByName(providerURLParser.getHostName());
        port = providerURLParser.getPort();
        this.providerURL = providerURL;
    }

    /**
     * @return the address of the server
     */
    public InetAddress getInetAddress() {
        return inetAddress;
    }

    /**
     * @return port of the provider URL
     */
    public int getPort() {
        return port;
    }

    @Override
    public String toString(){
        return providerURL;
    }

    /**
     * @return the protocol of this reference
     */
    public String getProtocol() {
        return protocol;
    }

    @Override
    public boolean equals(final Object obj) {
        if(obj == null || !(obj instanceof ServerRef)) {
            return false;
        }
        ServerRef serverRef = (ServerRef) obj;
        return inetAddress.equals(serverRef.inetAddress)
            && port == serverRef.port
            && (protocol == null || protocol.equals(serverRef.protocol));
    }

    @Override
    public int hashCode() {
        return inetAddress.hashCode() + port;
    }

    /**
     * @return a provider URL
     */
    public String getProviderURL() {
        return providerURL;
    }

    /**
     * @return the reference of server on which is deployed this object.
     */
    public ServerRef getServerRef() {
        return this;
    }

}
