/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:JNDIConfig.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.config;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;
import java.util.Map.Entry;

import javax.naming.Context;

import net.jcip.annotations.ThreadSafe;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Utility to configure JNDI when CMI is used in standalone.
 * The methods {@link #getCMIEnv()}, {@link #getCMIEnv(URL)} and {@link #getCMIEnv(Properties)}
 * allows to get a proper environment before creating a new {@link InitialContext}.
 * When {@link http://carol.objectweb.org Carol} is preferred this class must not be used.
 */
@ThreadSafe
public final class JNDIConfig {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(JNDIConfig.class);

    /**
     * Property name for defining the protocol.
     */
    public static final String WRAPPED_PROTOCOL = "cmi.context.wrapped.protocol";

    /**
     * Property name for defining the initial provider URLs.
     */
    public static final String INITIAL_PROVIDER_URLS = "cmi.context.provider.urls";

    /**
     * Property name for defining the wrapped InitialContextFactory.
     */
    public static final String WRAPPED_INITIAL_CONTEXT_FACTORY = "cmi.context.wrapped.factory";

    /**
     * Property name for specifying if we are in server mode.
     */
    public static final String SERVER_MODE = "cmi.mode.server";

    /**
     * Protocol used in the JVM.
     */
    private static volatile String protocol;

    /**
     * Utility so the default constructor is private.
     */
    private JNDIConfig() {}

    /**
     * Returns the JNDI environment to use CMI in standalone.
     * It will be extracted from the default configuration file: 'cmi.properties'.
     * @return the JNDI environment to use CMI in standalone
     * @throws CMIConfigException if the provided configuration file is not valid
     */
    public static synchronized Hashtable<String, ?> getCMIEnv() throws CMIConfigException {
        return getCMIEnv(Thread.currentThread().getContextClassLoader().getResource(CMIConfig.CMI_PROPERTIES));
    }

    /**
     * Returns the JNDI environment to use CMI in standalone.
     * It will be extracted from the given url.
     * @param url an URL to find the configuration file
     * @return the JNDI environment to use CMI in standalone
     * @throws CMIConfigException if the provided configuration file is not valid
     */
    public static synchronized Hashtable<String, ?> getCMIEnv(final URL url) throws CMIConfigException {
        return getCMIEnv(UtilConfig.getPropertiesFromURL(url));
    }

    /**
     * Returns the JNDI environment to use CMI in standalone.
     * It will be extracted from the given set of properties.
     * @param properties a set of properties
     * @return the JNDI environment to use CMI in standalone
     * @throws CMIConfigException if the provided configuration file is not valid
     */
    public static synchronized Hashtable<String, ?> getCMIEnv(final Properties properties) throws CMIConfigException {

        Hashtable<String, Object> cmiEnv = new Hashtable<String, Object>();
        for(Entry<Object, Object> e : properties.entrySet()) {
            String s = (String) e.getValue();
            Object k = e.getKey();
            if(k.equals(INITIAL_PROVIDER_URLS)) {
                try {
                    List<String>  providerURLs = new ArrayList<String>(ProviderURLsParser.parse(s));
                    cmiEnv.put(INITIAL_PROVIDER_URLS, providerURLs);
                } catch (MalformedURLException e1) {
                    LOGGER.error("Cannot parse provider urls {0}", s, e);
                    throw new CMIConfigException("Cannot parse provider urls "+s, e1);
                }
            } else if(k.equals(WRAPPED_INITIAL_CONTEXT_FACTORY)) {
                cmiEnv.put(WRAPPED_INITIAL_CONTEXT_FACTORY, s);
            } else if(k.equals(WRAPPED_PROTOCOL)) {
                protocol = s;
                LOGGER.debug("The protocol is {0}", protocol);
                cmiEnv.put(WRAPPED_PROTOCOL, s);
            } else if(k.equals(SERVER_MODE)) {
                cmiEnv.put(SERVER_MODE, Boolean.valueOf(s));
            } else {
                cmiEnv.put((String) k, s);
            }
        }
        cmiEnv.put(Context.INITIAL_CONTEXT_FACTORY, "org.ow2.carol.cmi.jndi.context.CMIInitialContextFactory");
        String proDelegate = System.getProperty("javax.rmi.CORBA.PortableRemoteObjectClass");
        LOGGER.debug("Current PRO delegate is {0}", proDelegate);
        if(proDelegate == null || proDelegate != null
                && proDelegate.equals("com.sun.corba.se.impl.javax.rmi.PortableRemoteObject")) {
            LOGGER.debug("Using the class 'org.ow2.carol.cmi.rmi.DefaultCMIPRODelegate' as PRO delegate");
            System.setProperty("javax.rmi.CORBA.PortableRemoteObjectClass",
                "org.ow2.carol.cmi.rmi.DefaultCMIPRODelegate");
        }
        if(protocol == "iiop") {
            LOGGER.debug("Using the class 'com.sun.corba.se.impl.presentation.rmi.StubFactoryFactoryStaticImpl'"
                    +"as ORBDynamicStubFactoryFactory");
            System.setProperty("com.sun.CORBA.ORBDynamicStubFactoryFactoryClass",
            "com.sun.corba.se.impl.presentation.rmi.StubFactoryFactoryStaticImpl");
        }
        return cmiEnv;
    }

    /**
     * @return the protocol used with CMI
     */
    public static String getProtocol() {
        return protocol;
    }

    /**
     * Sets the protocol used with CMI.
     * @param protocol a name of protocol
     */
    public static void setProtocol(final String protocol) {
        JNDIConfig.protocol = protocol;
    }

}
