/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: HAEJBHomeInvocationHandler.java 1333 2007-10-17 20:32:51Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.ha.rpc;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import javax.ejb.CreateException;
import javax.ejb.EJBHome;
import javax.ejb.EJBMetaData;
import javax.ejb.EJBObject;
import javax.ejb.FinderException;
import javax.ejb.Handle;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.controller.common.AbsClusterViewManager;
import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.ejb2_1.rpc.EJBInvocationHandlerException;
import org.ow2.carol.cmi.ejb2_1.spec.CMIHomeHandle;
import org.ow2.carol.cmi.ejb2_1.spec.JMetaData;
import org.ow2.carol.cmi.reference.CMIReferenceable;
import org.ow2.carol.cmi.reference.CMIReferenceableWrapper;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.carol.cmi.rpc.CMIInvocationHandler;
import org.ow2.carol.cmi.rpc.CMIProxy;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This class intercepts the invocations on the home interface of a stateful ejb2.
 * It is created by {@link org.ow2.carol.cmi.rpc.CMIProxyFactory#newCMIProxy(ClusterViewManager, String, String)}.
 * @author The new CMI team
 * @see org.ow2.carol.cmi.rpc.CMIProxyFactory
 */
@ThreadSafe
public class HAEJBHomeInvocationHandler extends CMIInvocationHandler {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(HAEJBHomeInvocationHandler.class);

    /**
     * The classloader to use.
     */
    private final ClassLoader classLoader;

    /**
     * Home interface.
     */
    private final Class<? extends EJBHome> homeClass;

    /**
     * Remote interface.
     */
    private final Class<? extends EJBObject> remoteClass;

    /**
     * The prefix for the name of a create method.
     */
    private static final String CREATE_METHOD_PREFIX = "create";

    /**
     * The prefix for the name of a find method.
     */
    private static final String FIND_METHOD_PREFIX = "find";

    /**
     * Keep in cache the metadata.
     */
    private EJBMetaData ejbMetaData = null;

    /**
     * Build a new EJB2 Invocation Handler for the home interface.
     * @param classLoader the classloader to use (when the smart factory is used, it is the smart classloader)
     * @param clusterViewManager A manager of the cluster view
     * @param objectName a name of the object
     * @param protocolName a protocol to perform the invocation
     * @param homeClass a home interface
     */
    public HAEJBHomeInvocationHandler(
            final ClassLoader classLoader,
            final ClusterViewManager clusterViewManager,
            final String objectName,
            final String protocolName,
            final Class<? extends EJBHome> homeClass) {
        super(clusterViewManager, objectName, protocolName, true);
        this.classLoader = classLoader;
        this.homeClass = homeClass;
        try {
            remoteClass = clusterViewManager.getRemoteClass(objectName);
        } catch (ObjectNotFoundException e) {
            throw new HAEJBInvocationHandlerException("Cannot get the remote class for "+objectName, e);
        }
    }

    /**
     * Returns a string representation for a proxy that uses this invocation
     * handler.
     **/
    @Override
    protected String proxyToString(final Object proxy) {
        return "HAEJBHomeProxy["
        + super.proxyToString(proxy)
        + ", homeClass:" + homeClass.getName()
        + ", remoteClass:" + remoteClass.getName()
        + ", ejbMetada:" + ejbMetaData
        + "]";
    }

    /**
     * Handles remote methods.
     **/
    @Override
    protected Object invokeRemoteMethod(final Object proxy, final Method method, final Object[] args)
    throws HAEJBInvocationHandlerException, Throwable {

        String methodName = method.getName();

        // Use the same classloader that during the creation of this object
        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        setClassLoader(oldClassLoader);
        try {
            if(methodName.startsWith(CREATE_METHOD_PREFIX) ||
                    methodName.startsWith(FIND_METHOD_PREFIX)) {
                EJBObject ejbObject =
                    (EJBObject) super.invokeRemoteMethod(proxy, method, args);
                CMIReferenceable<EJBObject> remoteStubWrapped =
                    new CMIReferenceableWrapper<EJBObject>(super.getCurrentRef().getReference(), ejbObject);
                super.setCurrentRef(null);
                // Return a proxy that handles the retrieved stub
                HAEJBObjectInvocationHandler ejbObjectInvocationHandler = new HAEJBObjectInvocationHandler(
                        classLoader, clusterViewManager, objectName,
                        protocolName, homeClass, remoteClass, (EJBHome) proxy,
                        AbsClusterViewManager.getClusterViewManager().getSessionId(),
                        new CreateMethodWrapper(method, args), remoteStubWrapped);
                Object cmiProxy = Proxy.newProxyInstance(
                        classLoader, new Class[] {remoteClass, CMIProxy.class}, ejbObjectInvocationHandler);
                return cmiProxy;
            } else if(methodName.equals("getEJBMetaData")) {
                if(ejbMetaData == null) {
                    EJBMetaData ejbMetaDataRaw = (EJBMetaData) super.invokeRemoteMethod(proxy, method, args);
                    ejbMetaData = new JMetaData(
                            (EJBHome) proxy, homeClass, remoteClass, ejbMetaDataRaw.isSession(),
                            ejbMetaDataRaw.isStatelessSession(), ejbMetaDataRaw.getPrimaryKeyClass());
                }
                return ejbMetaData;
            } else if(methodName.equals("getHomeHandle")) {
                return new CMIHomeHandle((EJBHome) proxy);
            } else if(methodName.equals("remove")) {
                ((Handle) args[0]).getEJBObject().remove();
                return null;
            } else {
                throw new NoSuchMethodException(method.toString());
            }
        } catch(Exception e) {
            // Don't catch the exceptions thrown by the container
            if(e instanceof CreateException || e instanceof FinderException) {
                throw e;
            }
            LOGGER.error("Error when invoking {0}", method, e);
            throw new EJBInvocationHandlerException("Error when invoking "+method, e);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    }

}
