/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: HAEJBObjectInvocationHandler.java 1333 2007-10-17 20:32:51Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.ha.rpc;

import java.lang.reflect.Method;

import javax.ejb.EJBException;
import javax.ejb.EJBHome;
import javax.ejb.EJBObject;

import org.ow2.carol.cmi.ha.interceptor.HACurrent;
import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.ejb2_1.rpc.EJBInvocationHandlerException;
import org.ow2.carol.cmi.ejb2_1.spec.CMIHandle;
import org.ow2.carol.cmi.ha.RequestId;
import org.ow2.carol.cmi.ha.SessionId;
import org.ow2.carol.cmi.reference.CMIReferenceable;
import org.ow2.carol.cmi.reference.CMIReferenceableWrapper;
import org.ow2.carol.cmi.rpc.CMIInvocationHandler;
import org.ow2.carol.cmi.rpc.CMIInvocationHandlerException;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This class intercepts the invocations on the remote interface of a stateful or entity ejb2.
 * It is created by {@link org.ow2.carol.cmi.rpc.CMIProxyFactory#newCMIProxy(ClusterViewManager, String, String)}.
 * @author The new CMI team
 * @see org.ow2.carol.cmi.rpc.CMIProxyFactory
 */
public class HAEJBObjectInvocationHandler extends CMIInvocationHandler {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(HAEJBObjectInvocationHandler.class);

    /**
     * The classloader to use.
     */
    private final ClassLoader classLoader;

    /**
     * Home interface.
     */
    private final Class<? extends EJBHome> homeClass;

    /**
     * Remote interface.
     */
    private final Class<? extends EJBObject> remoteClass;

    /**
     * Home proxy.
     */
    private final EJBHome ejbHomeProxy;

    /**
     * Identify the session associated with this proxy.
     */
    private final SessionId sessionId;

    /**
     * A number of request.
     */
    private int requestNb = 0;

    /**
     * A suffix for the name of the create method.
     */
    private final CreateMethodWrapper createMethodWrapper;

    /**
     * The wrapped remote stub currently used.
     */
    private CMIReferenceable<EJBObject> wrappedRemoteStub;

    /**
     * True if the proxy is on failover.
     */
    private boolean onFailover = false;

    /**
     * Build a new EJB2 Invocation Handler.
     * @param classLoader the classloader to use (when the smart factory is used, it is the smart classloader)
     * @param clusterViewManager A manager of the cluster view
     * @param objectName a name of the object
     * @param protocolName a protocol to perform the invocation
     * @param homeClass
     * @param remoteClass interface of the object
     * @param ejbHomeProxy
     */
    public HAEJBObjectInvocationHandler(
            final ClassLoader classLoader,
            final ClusterViewManager clusterViewManager,
            final String objectName,
            final String protocolName,
            final Class<? extends EJBHome> homeClass,
            final Class<? extends EJBObject> remoteClass,
            final EJBHome ejbHomeProxy,
            final SessionId sessionId,
            final CreateMethodWrapper createMethodWrapper,
            final CMIReferenceable<EJBObject> wrappedRemoteStub) {
        super(clusterViewManager, objectName, protocolName, true);
        this.classLoader = classLoader;
        this.homeClass = homeClass;
        this.remoteClass = remoteClass;
        this.ejbHomeProxy = ejbHomeProxy;
        this.sessionId = sessionId;
        this.createMethodWrapper = createMethodWrapper;
        this.wrappedRemoteStub = wrappedRemoteStub;
    }

    /**
     * Returns a string representation for a proxy that uses this invocation
     * handler.
     **/
    @Override
    protected String proxyToString(final Object proxy) {
        return "HAEJBObjectProxy["
        + super.proxyToString(proxy)
        + ", homeClass:" + homeClass.getName()
        + ", remoteClass:" + remoteClass.getName()
        + ", ejbHomeProxy:" + ejbHomeProxy
        + ", sessionId:" + sessionId
        + ", createMethodWrapper:" + createMethodWrapper
        + ", wrappedRemoteStub:" + wrappedRemoteStub
        + ", onFailover:" + onFailover
        + "]";
    }

    /**
     * Handles remote methods.
     **/
    @Override
    protected Object invokeRemoteMethod(final Object proxy, final Method method, final Object[] args)
    throws HAEJBInvocationHandlerException, Throwable {

        String methodName = method.getName();

        // Use the same classloader that during the creation of this object
        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        setClassLoader(oldClassLoader);
        try {
            if(methodName.equals("getEJBHome")) {
                return ejbHomeProxy;
            } else if(methodName.equals("getHandle")) {
                return new CMIHandle((EJBObject) proxy);
            } else if(methodName.equals("isIdentical")) {
                EJBObject ejbObject = (EJBObject) args[0];
                return homeClass.equals(ejbObject.getEJBHome().getEJBMetaData().getHomeInterfaceClass());
            }

            // Generate a requestId
            RequestId requestId = new RequestId(sessionId, requestNb);
            // Set this request in the current context
            HACurrent.getHACurrent().putNextReq(requestId);

            super.setCurrentRef(wrappedRemoteStub);

            Object result = super.invokeRemoteMethod(proxy, method, args);

            // Update the request number now that all is OK
            requestNb++;
            // Unset onFailover
            if (onFailover) {
                onFailover = false;
                HACurrent.getHACurrent().setOnFailover(false);
            }
            return result;
        } catch(Exception e) {
            if(e instanceof EJBException) {
                throw e;
            }
            LOGGER.error("Error when invoking {0}", method, e);
            throw new EJBInvocationHandlerException("Error when invoking "+method, e);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    }

    @Override
    protected void onExceptionHook() throws CMIInvocationHandlerException, Throwable {
        EJBObject ejbObject;
        try {
            ejbObject = (EJBObject) super.invokeRemoteMethod(
                    ejbHomeProxy, createMethodWrapper.getMethod(), createMethodWrapper.getArgs());
        } catch (Exception e) {
            if(e instanceof EJBException) {
                throw e;
            }
            LOGGER.error("Failover: Error when invoking {0}", createMethodWrapper.getMethod(), e);
            throw new EJBInvocationHandlerException(
                    "Failover: Error when invoking "+createMethodWrapper.getMethod(), e);
        }
        wrappedRemoteStub =
            new CMIReferenceableWrapper<EJBObject>(super.getCurrentRef().getReference(), ejbObject);
        super.setCurrentRef(wrappedRemoteStub);
        // Set onFailover
        onFailover = true;
        // Propagate onFailover status to the server
        HACurrent.getHACurrent().setOnFailover(true);
    }

}
