/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: HAEJBObjectInvocationHandler.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.ejb2_1.rpc.ha;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.List;

import javax.ejb.EJBException;
import javax.ejb.EJBHome;
import javax.ejb.EJBObject;

import org.ow2.carol.cmi.ha.interceptor.HACurrent;
import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.ejb2_1.rpc.EJBInvocationHandlerException;
import org.ow2.carol.cmi.ejb2_1.spec.EJBObjectHandle;
import org.ow2.carol.cmi.ha.RequestId;
import org.ow2.carol.cmi.ha.SessionId;
import org.ow2.carol.cmi.pool.StubOrProxyFactory;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.CMIReferenceable;
import org.ow2.carol.cmi.reference.RemoteCMIReferenceableWrapper;
import org.ow2.carol.cmi.rpc.CMIInvocationHandler;
import org.ow2.carol.cmi.rpc.CMIProxy;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.JPool;

/**
 * This class intercepts the invocations on the remote interface of a stateful or entity ejb2.
 * It is created by {@link org.ow2.carol.cmi.rpc.CMIProxyFactory#newCMIProxy(ClusterViewManager, String, String)}.
 * @author The new CMI team
 * @see org.ow2.carol.cmi.rpc.CMIProxyFactory
 */
public class HAEJBObjectInvocationHandler extends CMIInvocationHandler<EJBObject> {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = -3471150548023542467L;

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(HAEJBObjectInvocationHandler.class);

    /**
     * The classloader to use.
     */
    private transient ClassLoader classLoader;

    /**
     * Home interface.
     */
    private transient Class<? extends EJBHome> homeClass;

    /**
     * Home proxy.
     */
    private transient EJBHome ejbHomeProxy;

    /**
     * Identify the session associated with this proxy.
     */
    private SessionId sessionId;

    /**
     * A number of request.
     */
    private int requestNb = 0;

    /**
     * A suffix for the name of the create method.
     */
    private CreateMethodWrapper createMethodWrapper;

    /**
     * True if the proxy is on failover.
     */
    private boolean onFailover = false;


    /**
     * Build a new EJB2 Invocation Handler.
     * @param classLoader the classloader to use (when the smart factory is used, it is the smart classloader)
     * @param clusterViewManager A manager of the cluster view
     * @param objectName a name of the object
     * @param protocolName a protocol to perform the invocation
     * @param homeClass
     * @param remoteClass interface of the object
     * @param ejbHomeProxy
     */
    public HAEJBObjectInvocationHandler(
            final ClassLoader classLoader,
            final ClusterViewManager clusterViewManager,
            final String objectName,
            final String protocolName,
            final Class<? extends EJBHome> homeClass,
            final Class<? extends EJBObject> remoteClass,
            final EJBHome ejbHomeProxy,
            final SessionId sessionId,
            final CreateMethodWrapper createMethodWrapper,
            final CMIReferenceable<EJBObject> wrappedRemoteStub) {
        super(clusterViewManager, objectName, protocolName, true, remoteClass);
        this.classLoader = classLoader;
        this.homeClass = homeClass;
        this.ejbHomeProxy = ejbHomeProxy;
        this.sessionId = sessionId;
        this.createMethodWrapper = createMethodWrapper;
        setCurrentRef(wrappedRemoteStub);
    }

    /**
     * Returns a string representation for a proxy that uses this invocation
     * handler.
     **/
    @Override
    protected String proxyToString(final Object proxy) {
        return "HAEJBObjectProxy["
        + super.proxyToString(proxy)
        + ", homeClass:" + homeClass.getName()
        + ", ejbHomeProxy:" + ejbHomeProxy
        + ", sessionId:" + sessionId
        + ", requestNb:" + requestNb
        + ", createMethodWrapper:" + createMethodWrapper
        + ", onFailover:" + onFailover
        + "]";
    }

    /**
     * Handles remote methods.
     **/
    @Override
    protected Object invokeRemoteMethod(final Object proxy, final Method method, final Object... args)
    throws HAEJBInvocationHandlerException, Throwable {

        String methodName = method.getName();

        // Use the same classloader that during the creation of this object
        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        setClassLoader(oldClassLoader);
        try {
            if(methodName.equals("getEJBHome")) {
                return ejbHomeProxy;
            } else if(methodName.equals("isIdentical")) {
                EJBObject ejbObject = (EJBObject) args[0];
                return homeClass.equals(ejbObject.getEJBHome().getEJBMetaData().getHomeInterfaceClass());
            }

            // Generate a requestId
            RequestId requestId = new RequestId(sessionId, requestNb);
            // Set this request in the current context
            HACurrent.getHACurrent().putNextReq(requestId);

            Object result = super.invokeRemoteMethod(proxy, method, args);

            // Update the request number now that all is OK
            requestNb++;
            // Unset onFailover
            if (onFailover) {
                onFailover = false;
                HACurrent.getHACurrent().setOnFailover(false);
            }
            return result;
        } catch(Exception e) {
            if(e instanceof EJBException) {
                throw e;
            }
            LOGGER.error("Error when invoking {0}", method, e);
            throw new EJBInvocationHandlerException("Error when invoking " + method, e);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    }

    @Override
    protected void onExceptionHook() throws HAEJBInvocationHandlerException, Throwable {
        EJBObject ejbObject;
        List<Class<?>> argTypes = new ArrayList<Class<?>>();
        for(String argTypename : createMethodWrapper.getArgTypenames()) {
            argTypes.add(Class.forName(argTypename));
        }
        Method createMethod = homeClass.getDeclaredMethod(
                createMethodWrapper.getMethodName(), argTypes.toArray(new Class<?>[argTypes.size()]));
        try {

            ejbObject = (EJBObject) super.invokeRemoteMethod(
                    ejbHomeProxy, createMethod, createMethodWrapper.getArgs());
        } catch (Exception e) {
            if(e instanceof EJBException) {
                throw e;
            }
            LOGGER.error("Failover: Error when invoking {0}", createMethod, e);
            throw new HAEJBInvocationHandlerException(
                    "Failover: Error when invoking " + createMethod, e);
        }
        CMIReferenceable<EJBObject> wrappedRemoteStub =
            new RemoteCMIReferenceableWrapper(getCurrentRef().getReference(), ejbObject);
        setCurrentRef(wrappedRemoteStub);
        // Set onFailover
        onFailover = true;
        // Propagate onFailover status to the server
        HACurrent.getHACurrent().setOnFailover(true);
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void checkInitialized() throws HAEJBInvocationHandlerException {
        try {
            synchronized(clusterViewManager){
                // The first time, create a pool to share stubs for EJBHome
                if(clusterViewManager.getPool(objectName) == null) {
                    LOGGER.debug("First lookup on {0}: creation of the pool for its stubs for EJBHome", objectName);
                    clusterViewManager.setPool(
                            objectName,
                            new JPool<CMIReferenceable, CMIReference>(
                                    new StubOrProxyFactory(clusterViewManager)));
                }
            }
            if(classLoader == null) {
                classLoader = Thread.currentThread().getContextClassLoader();
            }
            if(homeClass == null) {
                homeClass = (Class<? extends EJBHome>) clusterViewManager.getInterface(objectName);
            }
            if(ejbHomeProxy == null) {
                // Use the HA EJB2 handler
                HAEJBHomeInvocationHandler invocationHandler = new HAEJBHomeInvocationHandler(
                        classLoader, clusterViewManager, objectName, protocolName, homeClass);
                ejbHomeProxy = (EJBHome) Proxy.newProxyInstance(
                        classLoader,  new Class<?>[] {homeClass, CMIProxy.class}, invocationHandler);
            }
        } catch (Exception e) {
            LOGGER.error("Cannot init the HA EJBObject proxy", e);
            throw new HAEJBInvocationHandlerException("Cannot init the HA EJBObject proxy", e);
        }
    }

    @Override
    protected EJBObjectHandle getHandle(final CMIProxy cmiProxy) {
        if(cmiProxyHandle == null) {
            cmiProxyHandle = new EJBObjectHandle(objectName, itf.getName(), (EJBObject) cmiProxy);
        }
        return (EJBObjectHandle) cmiProxyHandle;
    }

}
