/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: EJBObjectFactory.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.ejb2_1.pool;

import java.lang.reflect.Method;
import java.util.Hashtable;

import javax.ejb.EJBHome;
import javax.ejb.EJBObject;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.ejb2_1.rpc.EJBInvocationHandlerException;
import org.ow2.carol.cmi.pool.NamingPoolException;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.CMIReferenceable;
import org.ow2.carol.cmi.reference.RemoteCMIReferenceableWrapper;
import org.ow2.carol.cmi.reference.ServerRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.PoolEntryStatistics;
import org.ow2.util.pool.api.PoolException;
import org.ow2.util.pool.impl.PoolFactory;

/**
 * This is the pool factory to share stubs for {@link EJBObject}.
 * For stateless EJB2 only.
 * @author The new CMI team
 */
@SuppressWarnings("unchecked")
public class EJBObjectFactory implements PoolFactory<CMIReferenceable, CMIReference> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(EJBObjectFactory.class);

    /**
     * Delegate to access at the cluster view.
     */
    private final ClusterViewManager clusterViewManager;

    public EJBObjectFactory(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
    }

    @SuppressWarnings("unchecked")
    public CMIReferenceable create(final CMIReference cmiReference) throws PoolException {
        ServerRef serverRef = cmiReference.getServerRef();
        String protocol = serverRef.getProtocol();
        // Get a context to perform a lookup
        Hashtable<String, Object> env = new Hashtable<String, Object>();
        env.put(Context.INITIAL_CONTEXT_FACTORY, clusterViewManager.getInitialContextFactoryName(protocol));
        env.put(Context.PROVIDER_URL, serverRef.getProviderURL());
        Context chosenContext;
        try{
            chosenContext = new InitialContext(env);
        } catch (NamingException e) {
            LOGGER.error("Cannot get a real context", e);
            throw new PoolException("Cannot get a real context", e);
        }
        if(chosenContext == null) {
            LOGGER.error("Cannot get a real context");
            throw new PoolException("Cannot get a real context");
        }

        // Get a JNDI name to perform a lookup
        String bindName = cmiReference.getObjectName();
        Class<? extends EJBHome> homeClass;
        try {
            homeClass = (Class<? extends EJBHome>) clusterViewManager.getInterface(bindName);
        } catch (Exception e) {
            LOGGER.error("Cannot get interface for name {0}", bindName, e);
            throw new PoolException("Cannot get interface for name "+bindName, e);
        }

        // Lookup...
        EJBHome ejbHome = null;
        try {
            LOGGER.debug("Lookup {0} on {1}", bindName, serverRef.getProviderURL());
            ejbHome = (EJBHome) PortableRemoteObject.narrow(chosenContext.lookup(bindName), homeClass);
        } catch (NamingException e) {
            LOGGER.info("No ClientClusterViewProvider is bound with the name {0} at the url {1}",
                    bindName, serverRef.getProviderURL(), e);
            throw new NamingPoolException(
                    "No ClientClusterViewProvider is bound with the name "+bindName
                    +" at the url "+serverRef.getProviderURL(), e);
        }
        Method methodCreate;
        try {
            methodCreate = homeClass.getDeclaredMethod("create");
        } catch (Exception e) {
            LOGGER.error("Cannot get method create for home class: " + homeClass.getName(), e);
            throw new EJBInvocationHandlerException("Cannot get method create for home class: " + homeClass.getName(), e);
        }
        Object ejbObject;
        try {
            ejbObject = methodCreate.invoke(ejbHome);
        } catch (Exception e) {
            LOGGER.error("Cannot invoke method create for ejb home: " + ejbHome, e);
            throw new EJBInvocationHandlerException("Cannot invoke method create for ejb home: " + ejbHome, e);
        }
        return new RemoteCMIReferenceableWrapper<EJBObject>(cmiReference, (EJBObject) ejbObject);
    }

    public boolean isMatching(final CMIReferenceable object, final CMIReference clue) {
        LOGGER.debug("{0} == {1} ?", object.getReference(), clue);
        return object.getReference().equals(clue);
    }

    public void remove(final CMIReferenceable object) {
    }

    public boolean validate(final CMIReferenceable object, final PoolEntryStatistics stats) {
        return true;
    }

}
