/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: HAEJBHomeInvocationHandler.java 1578 2008-01-20 15:34:28Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.ejb2_1.rpc.ha;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBHome;
import javax.ejb.EJBMetaData;
import javax.ejb.EJBObject;
import javax.ejb.FinderException;
import javax.ejb.Handle;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.controller.common.AbsClusterViewManager;
import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.ejb2_1.rpc.EJBInvocationHandlerException;
import org.ow2.carol.cmi.ejb2_1.spec.EJBHomeHandle;
import org.ow2.carol.cmi.ejb2_1.spec.JMetaData;
import org.ow2.carol.cmi.pool.StubOrProxyFactory;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.CMIReferenceable;
import org.ow2.carol.cmi.reference.CMIReferenceableWrapper;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.carol.cmi.rpc.CMIInvocationHandlerByPool;
import org.ow2.carol.cmi.rpc.CMIProxy;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.JPool;

/**
 * This class intercepts the invocations on the home interface of a stateful ejb2.
 * It is created by {@link org.ow2.carol.cmi.rpc.CMIProxyFactory#newCMIProxy(ClusterViewManager, String, String)}.
 * @author The new CMI team
 * @see org.ow2.carol.cmi.rpc.CMIProxyFactory
 */
@ThreadSafe
public class HAEJBHomeInvocationHandler extends CMIInvocationHandlerByPool<EJBHome> {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = 4045108869657013960L;

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(HAEJBHomeInvocationHandler.class);

    /**
     * The classloader to use.
     */
    private transient ClassLoader classLoader;

    /**
     * Remote interface.
     */
    private transient Class<? extends EJBObject> remoteClass;

    /**
     * The prefix for the name of a create method.
     */
    private static final String CREATE_METHOD_PREFIX = "create";

    /**
     * The prefix for the name of a find method.
     */
    private static final String FIND_METHOD_PREFIX = "find";

    /**
     * Keep in cache the metadata.
     */
    private transient EJBMetaData ejbMetaData = null;


    /**
     * Build a new EJB2 Invocation Handler for the home interface.
     * @param classLoader the classloader to use (when the smart factory is used, it is the smart classloader)
     * @param clusterViewManager A manager of the cluster view
     * @param objectName a name of the object
     * @param protocolName a protocol to perform the invocation
     * @param homeClass a home interface
     */
    public HAEJBHomeInvocationHandler(
            final ClassLoader classLoader,
            final ClusterViewManager clusterViewManager,
            final String objectName,
            final String protocolName,
            final Class<? extends EJBHome> homeClass) {
        super(clusterViewManager, objectName, protocolName, true, homeClass);
        this.classLoader = classLoader;
        try {
            remoteClass = clusterViewManager.getRemoteClass(objectName);
        } catch (ObjectNotFoundException e) {
            throw new HAEJBInvocationHandlerException("Cannot get the remote class for " + objectName, e);
        }
    }

    /**
     * Returns a string representation for a proxy that uses this invocation
     * handler.
     **/
    @Override
    protected String proxyToString(final Object proxy) {
        return "HAEJBHomeProxy["
        + super.proxyToString(proxy)
        + ", remoteClass:" + remoteClass.getName()
        + ", ejbMetada:" + ejbMetaData
        + "]";
    }

    /**
     * Handles remote methods.
     **/
    @Override
    protected Object invokeRemoteMethod(
            final Object proxy, final Method method, final Object... args) throws Throwable {

        String methodName = method.getName();

        // Use the same classloader that during the creation of this object
        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        setClassLoader(oldClassLoader);
        try {
            if(methodName.startsWith(CREATE_METHOD_PREFIX) ||
                    methodName.startsWith(FIND_METHOD_PREFIX)) {
                CMIReferenceable<EJBObject> remoteStubWrapped = getWrappedEJBObject(proxy, method, args);

                // Cache ejbMetaData
                if(ejbMetaData == null) {
                    Method ejbMetaDataMethod = itf.getMethod("getEJBMetaData");
                    EJBMetaData ejbMetaDataRaw = (EJBMetaData) super.invokeRemoteMethod(proxy, ejbMetaDataMethod);
                    boolean session = ejbMetaDataRaw.isSession();
                    ejbMetaData = new JMetaData(
                            (EJBHome) proxy, itf, remoteClass, session,
                            ejbMetaDataRaw.isStatelessSession(), session? null : ejbMetaDataRaw.getPrimaryKeyClass());
                }

                // Don't use the same server at the next invocation
                setCurrentRef(null);

                // Prepare the method to use for failover
                Method createMethod;
                Object[] arguments;
                if(ejbMetaData.isSession()) {
                    // The same create method will be used before injecting the session
                    createMethod = method;
                    arguments = args;
                } else {
                    // Retrieve by finding the primary key
                    createMethod = remoteClass.getMethod("findByPrimaryKey", Object.class);
                    arguments = new Object[] {remoteStubWrapped.getReferencedObject().getPrimaryKey()};
                }

                List<String> argTypenames = new ArrayList<String>();
                for(Class<?> argType : createMethod.getParameterTypes()) {
                    argTypenames.add(argType.getName());
                }

                CreateMethodWrapper createMethodWrapper = new CreateMethodWrapper(
                        createMethod.getName(), argTypenames.toArray(new String[argTypenames.size()]), arguments);

                // Return a proxy that handles the retrieved stub
                HAEJBObjectInvocationHandler ejbObjectInvocationHandler = new HAEJBObjectInvocationHandler(
                        classLoader, clusterViewManager, objectName,
                        protocolName, itf, remoteClass, this, (EJBHome) proxy,
                        AbsClusterViewManager.getClusterViewManager().getSessionId(),
                        createMethodWrapper, remoteStubWrapped);
                Object cmiProxy = Proxy.newProxyInstance(
                        classLoader, new Class<?>[] {remoteClass, CMIProxy.class}, ejbObjectInvocationHandler);
                return cmiProxy;
            } else if(methodName.equals("getEJBMetaData")) {
                if(ejbMetaData == null) {
                    EJBMetaData ejbMetaDataRaw = (EJBMetaData) super.invokeRemoteMethod(proxy, method, args);
                    setCurrentRef(null);
                    boolean session = ejbMetaDataRaw.isSession();
                    ejbMetaData = new JMetaData(
                            (EJBHome) proxy, itf, remoteClass, session,
                            ejbMetaDataRaw.isStatelessSession(), session? null : ejbMetaDataRaw.getPrimaryKeyClass());
                }
                return ejbMetaData;
            } else if(methodName.equals("remove")) {
                ((Handle) args[0]).getEJBObject().remove();
                return null;
            } else {
                throw new NoSuchMethodException(method.toString());
            }
        } catch(Exception e) {
            // Don't catch the exceptions thrown by the container
            if(e instanceof CreateException || e instanceof FinderException) {
                throw e;
            }
            LOGGER.error("Error when invoking {0}", method, e);
            throw new EJBInvocationHandlerException("Error when invoking " + method, e);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void checkInitialized() throws HAEJBInvocationHandlerException {
        try {
            synchronized(clusterViewManager){
                // The first time, create a pool to share stubs for EJBHome
                if(clusterViewManager.getPool(objectName) == null) {
                    LOGGER.debug("First lookup on {0}: creation of the pool for its stubs for EJBHome", objectName);
                    clusterViewManager.setPool(
                            objectName,
                            new JPool<CMIReferenceable, CMIReference>(
                                    new StubOrProxyFactory(clusterViewManager)));
                }
            }
            if(classLoader == null) {
                classLoader = Thread.currentThread().getContextClassLoader();
            }
            if(remoteClass == null) {
                remoteClass = clusterViewManager.getRemoteClass(objectName);
            }
        } catch (Exception e) {
            LOGGER.error("Cannot init the HA EJBHome proxy", e);
            throw new HAEJBInvocationHandlerException("Cannot init the HA EJBHome proxy", e);
        }
    }

    @Override
    protected String getHandleMethodName() {
        return "getHomeHandle";
    }

    @Override
    protected EJBHomeHandle getHandle(final CMIProxy cmiProxy) {
        if(cmiProxyHandle == null) {
            cmiProxyHandle = new EJBHomeHandle(objectName, itf.getName(), (EJBHome) cmiProxy);
        }
        return (EJBHomeHandle) cmiProxyHandle;
    }

    public CMIReferenceableWrapper<EJBObject> getWrappedEJBObject(
            final Object proxy, final Method method, final Object... args) throws Throwable {
        return new CMIReferenceableWrapper<EJBObject>(
                getCurrentRef().getReference(),
                (EJBObject) super.invokeRemoteMethod(proxy, method, args));
    }

}
