/**
 * Copyright (C) 2002-2007 - Bull S.A.S.
 *
 * CMI : Cluster Method Invocation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:Console.java 949 2007-06-02 17:24:33Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.test;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.TreeMap;

/**
 * This class implements a generic console.
 * you can add tokens and the associated actions to perform with
 * this class was developed in order to help the development
 *
 * @author Duvauchel Nicolas
 * @author The new CMI team
 *
 */

public class Console extends Thread {
    /**
     * Console name.
     */
    private String name;

    /**
     * List of the tokens.
     */
    private TreeMap<String, TokenInfos> listToken = new TreeMap<String, TokenInfos>();

    /**
     * InputStream.
     */
    private InputStreamReader is = new InputStreamReader(System.in);

    /**
     * Input Buffer.
     */
    private BufferedReader in = new BufferedReader(is);

    /**
     * The command line typed.
     */
    private String cmdLine;

    /**
     * The command line cut in words.
     */
    private String[] cmdWords;

    /**
     * Constructor.
     * @param name of the console
     */
    public Console(final String name) {
        this.name = name;
        addToken("quit", 0, null, "leaves the console!");
    }

    /**
     * this interface defines the interface that token handler class must implement.
     *
     */
    public interface TokenHandlerITF {
        void setArgs(final String[] args);
        int handle();
    }

    /**
     * This class represents the information linked with a token:
     * number of arguments, the handler, help information
     *
     */
    private final class TokenInfos {
        /**
         * Number of args of the token.
         */
        private int nbArgs;
        /**
         * Token handler.
         */
        private TokenHandlerITF handler;
        /**
         * Some help or description for the token.
         */
        private String help;

        /**
         * Constructor.
         * @param nbArgs
         * @param handler
         * @param help
         */
        private TokenInfos(final int nbArgs, final TokenHandlerITF handler, final String help) {
            this.nbArgs = nbArgs;
            this.handler = handler;
            this.help = help;
        }
    }

    /**
     * Display the help assoiated with the token.
     *
     */
    private void displayHelp() {
        for(String tok : listToken.keySet()) {
            TokenInfos tokInf = listToken.get(tok);
            System.out.println(" - " + tok + ": " + tokInf.help);
        }
    }

    /**
     * Add a token to the console.
     * @param tok : the token
     * @param nbArgs : the number of args
     * @param handler : the handler class
     * @param help : some help
     */
    public void addToken(final String tok, final int nbArgs, final TokenHandlerITF handler,
            final String help) {
        TokenInfos tokInf = new TokenInfos(nbArgs, handler, help);
        listToken.put(tok, tokInf);
    }

    /**
     * Run the console thread.
     */
    public void run() {
        System.out.println("Starting " + name + "...");

        do {

            System.out.print(name + ">");
            try {
                cmdLine = in.readLine();
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            cmdWords = cmdLine.split(" ");

            if (cmdWords[0].compareTo("?") == 0) {
                displayHelp();
            } else {
                TokenInfos tokInf = listToken.get(cmdWords[0]);
                if (tokInf == null) {
                    System.out.println("Unknown command");
                } else if (tokInf.nbArgs != (cmdWords.length - 1)) {
                    System.out.println("Wrong number of arguments:"
                            + (cmdWords.length - 1) + " instead of "
                            + tokInf.nbArgs);
                } else if (tokInf.handler != null) {
                    tokInf.handler.setArgs(cmdWords);
                    tokInf.handler.handle();
                }
            }

        } while (cmdWords[0].compareToIgnoreCase("quit") != 0);

    }
}
