/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:IIOPHelloServer.java 966 2007-06-05 07:54:33Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.test;

import java.io.FileOutputStream;
import java.io.PrintStream;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameClassPair;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.ow2.carol.cmi.admin.CMIAdmin;
import org.ow2.carol.cmi.annotation.Cluster;
import org.ow2.carol.cmi.annotation.LBPolicy;
import org.ow2.carol.cmi.config.JNDIConfig;
import org.ow2.carol.cmi.info.CMIInfoExtractor;
import org.ow2.carol.cmi.info.CMIInfoExtractorException;
import org.ow2.carol.cmi.info.ClusteredObjectInfo;
import org.ow2.carol.cmi.jndi.ClusteredObject;
import org.ow2.carol.cmi.lb.policy.RoundRobinPolicy;

/**
 * @author nieuviar
 * @author The new CMI team
 *
 */
@Cluster(name="test_cluster")
@LBPolicy(RoundRobinPolicy.class)
public class IIOPHelloServer extends PortableRemoteObject implements TestItf, ClusteredObject {

    private static ClusteredObjectInfo clusteredObjectInfo;

    private static CMIAdmin cmiAdmin = null;

    private static Context ic;

    private String msg;

    protected static class DisplayRegistryHandler implements Console.TokenHandlerITF {

        private String args[];

        public DisplayRegistryHandler() {
        }

        public void setArgs(String args[]) {
            this.args = args;
        }

        public int handle() {
            displayRegistry(ic);
            return 0;
        }
    }

    protected static class DisplayReplicatedRegistryHandler implements Console.TokenHandlerITF {

        private String args[];

        public DisplayReplicatedRegistryHandler() {
        }

        public void setArgs(String args[]) {
            this.args = args;
        }

        public int handle() {
            System.out.println();
            System.out.println("/*************************/");
            System.out.println("/* REPLICATED REGISTRY:");
            try {
                Set<String> clusterNames = cmiAdmin.getClusterNames();
                for(String clusterName : clusterNames) {
                    System.out.println();
                    System.out.println("/* In the cluster "+clusterName+":");
                    Set<String> objectNames = cmiAdmin.getObjectNames(clusterName);
                    for(String objectName : objectNames) {
                        System.out.println("/*   "+objectName+" has:");
                        System.out.println("/*  - instance(s) in the following nodes:");
                        List<String> jndiRefs = cmiAdmin.getServerRefs(objectName);
                        for(String jndiRef : jndiRefs) {
                            System.out.println("/*     >"+jndiRef);
                        }
                        System.out.println("/*  - for LB policy:");
                        System.out.println("/*     >"+cmiAdmin.getLBPolicyClassName(objectName));
                    }
                }
            } catch (Exception e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            System.out.println("/*************************/");
            System.out.println();
            return 0;
        }
    }

    public IIOPHelloServer(String string) throws RemoteException {
        msg = string;
        try {
            clusteredObjectInfo =
                CMIInfoExtractor.extractClusteringInfoFromAnnotatedPOJO(
                        TestItf.class, IIOPHelloServer.class, false, false, null);
        } catch (CMIInfoExtractorException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    private static void displayRegistry(Context ctx) {
        System.out.println();
        System.out.println("/********************/");
        System.out.println("/* REGISTRY:");
        try {
            NamingEnumeration<NameClassPair> list = ctx.list(ctx.getNameInNamespace());

            while (list.hasMore()) {
                NameClassPair current = list.next();
                String name = current.getName();
                System.out.println("/*    - " + name);
                System.out.println("/*  Servers which own an instance of this:");
                List<String> jndiRefs = cmiAdmin.getServerRefs(name, "iiop");
                for(String jndiRef:jndiRefs) {
                    System.out.println("/*  > "+jndiRef.toString());
                }
            }

            System.out.println("/********************/");
            System.out.println();

        } catch (NamingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

    }

    protected static class BindHandler implements Console.TokenHandlerITF {
        private String args[];

        public BindHandler() {
        }

        public BindHandler(String args[]) {
            this.args = args;
        }

        public void setArgs(String args[]) {
            this.args = args;
        }

        public int handle() {
            try {
                System.out.print("Binding \"" + args[1] + "\"...");
                IIOPHelloServer s = new IIOPHelloServer(args[1]);
                ic.bind(args[1], s);
                System.out.println("OK");
            } catch (NamingException e) {
                e.printStackTrace();
            } catch (RemoteException e) {
                e.printStackTrace();
            }
            return 0;
        }
    }

    protected static class RebindHandler implements Console.TokenHandlerITF {
        private String args[];

        public RebindHandler() {
        }

        public RebindHandler(String args[]) {
            this.args = args;
        }

        public void setArgs(String args[]) {
            this.args = args;
        }

        public int handle() {
            try {
                System.out.print("Rebinding \"" + args[1] + "\"...");
                IIOPHelloServer s = new IIOPHelloServer(args[1]);
                ic.rebind(args[1], s);
                System.out.println("OK");
            } catch (NamingException e) {
                e.printStackTrace();
            } catch (RemoteException e) {
                e.printStackTrace();
            }
            return 0;
        }
    }

    protected static class UnbindHandler implements Console.TokenHandlerITF {
        private String args[];

        public UnbindHandler() {
        }

        public UnbindHandler(String args[]) {
            this.args = args;
        }

        public void setArgs(String args[]) {
            this.args = args;
        }

        public int handle() {
            try {
                System.out.print("Unbinding \"" + args[1] + "\"...");
                ic.unbind(args[1]);
                System.out.println("OK");
            } catch (NamingException e) {
                // TODO Auto-generated catch block
                System.out.print("Naming exception");
                e.printStackTrace();
                return 1;
            }
            return 0;
        }
    }


    public static void main(final String[] args) throws Exception {

        FileOutputStream fos = new FileOutputStream("output/reports/errors-iiop.txt");
        PrintStream ps = new PrintStream(fos);
        System.setErr(ps);

        URL url = Thread.currentThread().getContextClassLoader().getResource("server-iiop.properties");

        Hashtable<String, ?> env = JNDIConfig.getCMIEnv(url);

        ic = new InitialContext(env);

        cmiAdmin = CMIAdmin.getCMIAdmin();

        Console cons = new Console("server");

        cons.addToken("dispreg", 0, new DisplayRegistryHandler(), "displays the registry");

        cons.addToken("bind", 1, new BindHandler(), "bind with the name [arg]");

        cons.addToken("rebind", 1, new RebindHandler(), "rebind with the name [arg]");

        cons.addToken("unbind", 1, new UnbindHandler(), "unbind the name [arg]");

        cons.addToken("disprep", 0, new DisplayReplicatedRegistryHandler(), "displays the replicated registry");

        cons.start();
    }

    /**
     * Displays "hello".
     * @throws RemoteException if a {@link RemoteException} occurs.
     **/
    public void display() throws RemoteException {
        System.out.println("Hello "+msg);
    }

    public ClusteredObjectInfo getClusteredObjectInfo() {
        return clusteredObjectInfo;
    }
}
