/**
* High Availability Service (HA) for JOnAS
*
* Copyright (C) 2006 Distributed Systems Lab.
* Universidad Polit??cnica de Madrid (Spain)
* Contact: http://lsd.ls.fi.upm.es/lsd
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
* USA
*
* --------------------------------------------------------------------------
* $Id: HaMessageData.java 1294 2007-10-07 11:24:11Z loris $
* --------------------------------------------------------------------------
*/
package org.ow2.carol.cmi.ha;

import java.io.Serializable;
import java.util.Vector;

/**
 * Holds the required information for the replication protocol.
 * @author Francisco Perez-Sorrosal (fpsorrosal@no-spam@fi.upm.es)
 * @author Alberto Paz-Jimenez (apaz@no-spam@fi.upm.es)
 */
public final class HaMessageData implements Serializable {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = -1656256040790087473L;

    /**
     * Kind of messages.
     */
    public enum HaMessageKind {
        /**
         * A commit is currently performed.
         */
        COMMITTING_MESSAGE,
        /**
         * A commit was performed.
         */
        COMMIT_MESSAGE,
        /**
         * An abort was performed.
         */
        ABORT_MESSAGE;
    }

    /**
     * message type.
     */
    private HaMessageKind type;

    /**
     * the request id.
     */
    private RequestId requestId;

    /**
     * the response.
     */
    private Object response;

    /**
     * the beans' state.
     */
    private Vector<BeanInfo> beans;

    /**
     * read-only.
     */
    private boolean readOnly;

    /**
     * other messages to be piggy-backed.
     */
    private Vector<HaMessageData> otherMessages;

    /**
     * Creates a new committing message.
     * @param requestId the request id
     * @param response the response
     * @param beans the beans
     * @param readOnly
     * @param otherMessages other messages
     * @return a new committing message
     */
    public static HaMessageData newCommittingMessage(final RequestId requestId,
                                        final Object response, final Vector<BeanInfo> beans,
                                        final boolean readOnly, final Vector<HaMessageData> otherMessages) {
        return new HaMessageData(HaMessageKind.COMMITTING_MESSAGE, requestId, response,
                                    beans, readOnly, otherMessages);
    }

    /**
     * Creates a new commit message.
     * @param requestId the request id
     * @return a new commit message
     */
    public static HaMessageData newCommitMessage(final RequestId requestId) {
        return new HaMessageData(HaMessageKind.COMMIT_MESSAGE, requestId, null, null, false, null);
    }

    /**
     * Creates a new abort message.
     * @param requestId the request id
     * @param response the response
     * @return a new abort message
     */
    public static HaMessageData newAbortMessage(final RequestId requestId, final Object response) {
        return new HaMessageData(HaMessageKind.ABORT_MESSAGE, requestId, response, null, false, null);
    }

    /**
     * Constructor.
     * @param type the type
     * @param requestId the request id
     * @param response the response
     * @param beans the beans
     * @param readOnly
     * @param otherMessages other messages
     */
    private HaMessageData(final HaMessageKind type, final RequestId requestId,
                            final Object response, final Vector<BeanInfo> beans,
                            final boolean readOnly, final Vector<HaMessageData> otherMessages) {
        this.type = type;
        this.requestId = requestId;
        this.response = response;
        this.beans = beans;
        this.readOnly = readOnly;
        this.otherMessages = otherMessages;
    }

    /**
     * @return Returns the beans.
     */
    public Vector<BeanInfo> getBeans() {
        return beans;
    }

    /**
     * @return Returns the readOnly
     */
    public boolean isReadOnly() {
        return readOnly;
    }

    /**
     * @return Returns the otherMessages.
     */
    public Vector<HaMessageData> getOtherMessages() {
        return otherMessages;
    }

    /**
     * @return Returns the requestId.
     */
    public RequestId getRequestId() {
        return requestId;
    }

    /**
     * @return Returns the response.
     */
    public Object getResponse() {
        return response;
    }

    /**
     * @return Returns the type.
     */
    public HaMessageKind getType() {
        return type;
    }

    @Override
    public String toString() {
        String ret = "\nRequest id: " + requestId + "\n";
        ret += "\ttype: " + type + "\n";

        if (beans != null) {
            ret += "\tBean changes:\n";
            for(BeanInfo bi : beans) {
                ret += "\t\t" + bi.toString() + "\n";
            }
        }

        if (response != null) {
            ret += "\tResponse: " +  response + "\n";
        }

        ret += "\treadOnly: " + readOnly;

        return ret;
    }
}
