/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:JGroupsClusterViewManager.java 914 2007-05-25 16:48:16Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.controller.server.impl.jgroups;

import java.io.Serializable;

import java.net.InetAddress;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.Vector;

import javax.management.MBeanServer;

import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

import org.jgroups.Address;
import org.jgroups.JChannel;
import org.jgroups.jmx.JmxConfigurator;
import org.jgroups.stack.IpAddress;

import org.ow2.carol.cmi.admin.MBeanUtils;
import org.ow2.carol.cmi.config.CMIConfig;
import org.ow2.carol.cmi.controller.common.CMIThreadFactory;
import org.ow2.carol.cmi.controller.server.AbsServerClusterViewManager;
import org.ow2.carol.cmi.controller.server.DistributedObjectInfo;
import org.ow2.carol.cmi.controller.server.ServerClusterViewManagerException;
import org.ow2.carol.cmi.controller.server.impl.jgroups.SynchronizedDistributedTree.DistributedTreeListener;
import org.ow2.carol.cmi.controller.server.impl.jgroups.SynchronizedDistributedTree.ViewListener;
import org.ow2.carol.cmi.lb.data.LBPolicyData;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Manager implemented with JGroups that contains CMI logic for the server-side.
 * @author The new CMI team
 * @see DistributedTree
 */
@ThreadSafe
public final class JGroupsClusterViewManager extends AbsServerClusterViewManager
implements DistributedTreeListener, ViewListener {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(JGroupsClusterViewManager.class);

    /**
     * Distributed tree from JGroups.
     */
    private final SynchronizedDistributedTree distributedTree;

    /**
     * Local address of the channel.
     */
    private final IpAddress localAddress;

    /**
     * A set of paths to delete by the dedicated thread.
     */
    @GuardedBy("PATHS_TO_DELETE")
    private static final Set<String> PATHS_TO_DELETE = new HashSet<String>();

    /**
     * A daemon to clean the distributedTree.
     */
    private final Thread garbageThread;

    /**
     * The handle to resume the execution of the garbage thread.
     */
    private final ResumableRunnable resumableRunnable;

    /**
     * A mutex.
     */
    private final Object lock = new Object();

    /**
     * Construct a JGroupsClusterViewManager.
     * @throws Exception if the DistributedTree cannot be instanced or started
     */
    private JGroupsClusterViewManager() throws Exception {

        // Gets the group name
        String groupName = JGroupsServerConfig.getGroupName();
        LOGGER.debug("Group name: {0}", groupName);

        String stack = JGroupsServerConfig.getStack();
        LOGGER.debug("The stack is :{0}", stack);

        // Create a new DistributedTree
        distributedTree = new SynchronizedDistributedTree(groupName, stack);

        // Add this object as a DistributedTreeListener
        distributedTree.addDistributedTreeListener(this);

        // Add this object as a ViewListener
        distributedTree.addViewListener(this);

        // Start the DistributedTree
        distributedTree.start();

        JChannel channel = (JChannel) distributedTree.getChannel();

        // Register the JMX MBeans of JGroups stack
        try {
            JmxConfigurator.registerChannel(channel,
                    MBeanUtils.getMBeanServer(), MBeanUtils.getMBeanDomainName(),
                    channel.getClusterName() + ",name=" + CMIConfig.getCMIAdminMBeanName(), true);
        } catch (Exception e) {
            LOGGER.warn("Unable to register the channel to the MBean Server", e);
        }

        // Get local address
        Address obj = distributedTree.getLocalAddress();
        if(obj == null) {
            LOGGER.error("Cannot get the local address");
            throw new JGroupsClusterViewManagerException("Cannot get the local address");
        } else if(obj instanceof IpAddress) {
            localAddress = (IpAddress) obj;
            LOGGER.debug("IpAddress: {0}", localAddress);
        } else {
            String className = obj.getClass().getName();
            LOGGER.error("At this time only IP address are supported, and not: {0}.", className);
            throw new JGroupsClusterViewManagerException(
                    "At this timen only IP address are supported, and not: " + className);
        }

        // Creates the thread that will clean the distributed tree
        CMIThreadFactory cmiThreadFactory = getCmiThreadFactory();
        resumableRunnable = new ResumableRunnable(){
            private volatile boolean suspended;
            private volatile boolean stopped;
            public void run() {
                suspended = false;
                stopped = false;
                while(!stopped) {
                    // Execute every 60 seconds
                    synchronized (this) {
                        try {
                            while (suspended && !stopped) {
                                wait();
                            }
                        } catch (InterruptedException e) {
                            LOGGER.debug("Garbage collector interrupted", e);
                        }
                    }
                    LOGGER.debug("Cleaning the distributed tree...");
                    synchronized (PATHS_TO_DELETE) {
                        Iterator<String> it = PATHS_TO_DELETE.iterator();
                        while(it.hasNext()) {
                            String pathToDelete = it.next();
                            removePath(pathToDelete);
                            it.remove();
                        }
                        suspended = true;
                    }
                }
                LOGGER.debug("Garbage collector stopped");
            }

            public void resumeExecution() {
                LOGGER.debug("Garbage collector resumed");

                synchronized (this) {
                    suspended = false;
                    notify();
                }
            }

            public void stopExecution() {
                LOGGER.debug("Garbage collector stopping...");

                synchronized (this) {
                    stopped = true;
                    notify();
                }
            }

        };
        garbageThread = cmiThreadFactory.newThread(resumableRunnable);
    }

    /**
     * Private factory of JGroupsClusterViewManager because it is for an internal use.
     * This method should be called only by {@link AbsServerClusterViewManager#getServerClusterViewManager()}, with reflection.
     * @return an instance of JGroupsClusterViewManager
     * @throws Exception if the DistributedTree cannot be instanced or started
     * @see AbsServerClusterViewManager#getServerClusterViewManager()
     */
    @SuppressWarnings("unused")
    private static JGroupsClusterViewManager getJGroupsClusterViewManager() throws Exception {
        JGroupsClusterViewManager jgroupsClusterViewManager = new JGroupsClusterViewManager();
        jgroupsClusterViewManager.garbageThread.start();
        return jgroupsClusterViewManager;
    }

    /**
     * Start the server-side manager.
     */
    @Override
    public void doStart() {
        // At this time, do nothing
    }

    /**
     * Stop the server-side manager.
     */
    @Override
    public void doStop() {
        resumableRunnable.stopExecution();
        String domain = MBeanUtils.getMBeanDomainName();
        JChannel channel = (JChannel) distributedTree.getChannel();
        String cluster_name = channel.getClusterName();
        String name = domain + ":type=channel,cluster=" + cluster_name + ",name=" + CMIConfig.getCMIAdminMBeanName();
        MBeanServer mBeanServer = MBeanUtils.getMBeanServer();
        try {
            JmxConfigurator.unregisterChannel(mBeanServer, name);
        } catch (Exception e) {
            LOGGER.error("Error when unregistering the channel with name {0} from the MBean server", cluster_name, e);
        }
        String tmp = domain + ":type=protocol,cluster=" + cluster_name + ",name=" + CMIConfig.getCMIAdminMBeanName();
        JmxConfigurator.unregisterProtocols(mBeanServer, channel, tmp);
        distributedTree.stop();
    }

    /**
     * Removes the given path from the distributedTree.
     * @param pathToDelete a path in the distributedTree
     */
    private void removePath(final String pathToDelete) {
        distributedTree.remove(pathToDelete);
    }

    /**
     * Returns true if the given object is already replicated.
     * @param objectName a name of object
     * @return true if the given object is already replicated
     */
    @Override
    protected boolean containObject(final String objectName) {
        return distributedTree.exists(getObjectPath(objectName));
    }

    /**
     * Returns informations on this object.
     * @param objectName a name of object
     * @return informations on this object
     * @throws ObjectNotFoundException if the given object is not found
     */
    @Override
    protected DistributedObjectInfo getDistributedObjectInfo(final String objectName)
    throws ObjectNotFoundException {
        String objectPath = getObjectPath(objectName);
        Object obj = distributedTree.get(objectPath);
        if(obj == null) {
            LOGGER.error("Unknown object with name {0}", objectName);
            throw new ObjectNotFoundException("Unknown object with name "+objectName);
        }
        try {
            return ((DistributedObjectInfo) obj).clone();
        } catch (CloneNotSupportedException e) {
            throw new Error("Amazing bug... I don't believe it !");
        }
    }

    /**
     * Sets informations on the clustered object with the given name.
     * @param objectName a name of object
     * @param distributedObjectInfo informations on the clustered object
     */
    @Override
    protected void setDistributedObjectInfo(final String objectName,
            final DistributedObjectInfo distributedObjectInfo) {
        String objectPath = getObjectPath(objectName);
        LOGGER.debug("Adding {0} at the path {1}", distributedObjectInfo, objectPath);
        LOGGER.debug("Old infos: {0}", distributedTree.get(objectPath));
        distributedTree.reset(objectPath, distributedObjectInfo);
    }

    /**
     * Registers an association between an address and a CMIReference.
     * @param cmiReference a CMIReference
     */
    private void registerPath(final CMIReference cmiReference) {
        String path = getCMIReferencePath(cmiReference);
        String hostname = localAddress.getIpAddress().getHostAddress();
        int port = localAddress.getPort();
        String address = getAddressPath(hostname, port);
        String instAddr = getInstAddrPath(address, cmiReference.getServerRef().getProtocol()+"_"+cmiReference.getObjectName());
        distributedTree.add(instAddr, path);
    }

    /**
     * Unregisters an association between an address and a CMIReference.
     * @param cmiReference a CMIReference
     */
    private void unRegisterPath(final CMIReference cmiReference) {
        String hostname = localAddress.getIpAddress().getHostAddress();
        int port = localAddress.getPort();
        String address = getAddressPath(hostname, port);
        String instAddr = getInstAddrPath(address, cmiReference.getServerRef().getProtocol()+"_"+cmiReference.getObjectName());
        distributedTree.remove(instAddr);
    }

    /**
     * @param protocolName a name of protocol
     * @return the set of references on server connected to this server
     * @throws ServerClusterViewManagerException if the given protocol name doesn't exist
     */
    public Set<ServerRef> getServerRefs(final String protocolName) throws ServerClusterViewManagerException {
        Set<ServerRef> serverRefs = new HashSet<ServerRef>();
        for(String objectName : getObjectNames()) {
            try {
                for(CMIReference cmiRef : getCMIReferences(objectName, protocolName)) {
                    serverRefs.add(cmiRef.getServerRef());
                }
            } catch (ObjectNotFoundException e) {
                throw new ServerClusterViewManagerException(protocolName+" is unknown", e);
            }
        }
        return serverRefs;
    }

    /**
     * Returns a list of CMIReference for an object with the given name and protocol.
     * @param objectName a name of object
     * @param protocolName a name of protocol
     * @return a list of CMIReference for an object with the given name and protocol
     * @throws ObjectNotFoundException if none object has the given name for the given protocol
     */
    public List<CMIReference> getCMIReferences(final String objectName, final String protocolName)
    throws ObjectNotFoundException {
        String objectPath = getObjectPath(objectName);
        String protoPath = getProtocolPath(objectPath, protocolName);
        ArrayList<CMIReference> cmiReferences = new ArrayList<CMIReference>();
        Vector<String> childrenNames = distributedTree.getChildrenNames(protoPath);
        if(childrenNames.isEmpty()) {
            LOGGER.error("Unknown object {0} or protocol {1}", objectName, protocolName);
            throw new ObjectNotFoundException("Unknown object "+objectName+" or protocol "+protocolName);
        }
        for(String childrenName : childrenNames) {
            CMIReference cmiReference = getCMIReference(protoPath, childrenName);
            if(!isServerBlackListed(cmiReference.getServerRef())) {
                cmiReferences.add(cmiReference);
            } else {
                LOGGER.debug("Do not add {0} because the server with ref {1} is black listed",
                        cmiReference, cmiReference.getServerRef());
            }
        }
        return cmiReferences;
    }

    /**
     * Returns a list of CMIReference for an object with the given name.
     * @param objectName a name of object
     * @return a list of CMIReference for an object with the given name
     * @throws ObjectNotFoundException if no object is bound with the given name
     */
    public List<CMIReference> getCMIReferences(final String objectName)
    throws ObjectNotFoundException {
        String objectPath = getObjectPath(objectName);
        ArrayList<CMIReference> cmiReferences = new ArrayList<CMIReference>();
        Vector<String> protocols = distributedTree.getChildrenNames(objectPath);
        if(protocols.isEmpty()) {
            LOGGER.error("Unknown object {0}", objectName);
            throw new ObjectNotFoundException("Unknown object "+objectName);
        }
        for(String p : protocols) {
            String protoPath = getProtocolPath(objectPath, p);
            Vector<String> instances = distributedTree.getChildrenNames(protoPath);
            for(String nodeName : instances) {
                cmiReferences.add(getCMIReference(protoPath, nodeName));
            }
        }
        return cmiReferences;
    }

    /**
     * Return the reference for a given path for protocols and node name.
     * @param protocolPath the path for protocols
     * @param nodeName a node name
     * @return the reference for a given protocol and node
     */
    private CMIReference getCMIReference(final String protocolPath, final String nodeName) {
        String cmiReferencePath = getCMIReferencePath(protocolPath, nodeName);
        return (CMIReference) distributedTree.get(cmiReferencePath);
    }

    /**
     * Adds a CMIReference to the cluster view.
     * @param cmiReference a reference on an instance
     */
    @Override
    protected void addCMIReference(final CMIReference cmiReference) {
        String path = getCMIReferencePath(cmiReference);
        distributedTree.add(path, cmiReference);
        registerPath(cmiReference);
    }

    /**
     * Removes a CMIReference from the cluster view.
     * @param cmiReference a reference on a clustered object
     */
    public void removeCMIReference(final CMIReference cmiReference) {
        LOGGER.debug("Removing {0}...", cmiReference);
        String objectName = cmiReference.getObjectName();
        String objectPath = getObjectPath(objectName);
        String refPath = getCMIReferencePath(cmiReference);
        String protocolPath = getProtocolPath(objectPath, cmiReference.getServerRef().getProtocol());
        distributedTree.remove(refPath);
        unRegisterPath(cmiReference);
        Vector<?> childrenNames = distributedTree.getChildrenNames(protocolPath);
        if(childrenNames.isEmpty()) {
            distributedTree.remove(objectPath);
        }
    }

    /**
     * Returns the set of name of cluster.
     * @return a set of name of cluster
     */
    public Set<String> getClusterNames() {
        HashSet<String> clusterNames = new HashSet<String>();
        Vector<String> objectNames = distributedTree.getChildrenNames(getObjectsPath());
        DistributedObjectInfo distributedObjectInfo;
        for(String objectName : objectNames) {
            try {
                distributedObjectInfo = getDistributedObjectInfo(objectName);
                clusterNames.add(distributedObjectInfo.getClusterName());
            } catch (ObjectNotFoundException e) {
                throw new ServerClusterViewManagerException(objectName+" is missing !", e);
            }
        }
        return clusterNames;
    }

    /**
     * Returns the set of name of object for a given name of cluster.
     * @param clusterName a name of cluster
     * @return the set of name of object for a given name of cluster
     */
    public Set<String> getObjectNames(final String clusterName) {
        HashSet<String> objectNamesInCluster = new HashSet<String>();
        Vector<String> objectNames = distributedTree.getChildrenNames(getObjectsPath());
        DistributedObjectInfo distributedObjectInfo;
        for(String objectName : objectNames) {
            try {
                distributedObjectInfo = getDistributedObjectInfo(objectName);
                if(distributedObjectInfo.getClusterName().equals(clusterName)) {
                    objectNamesInCluster.add(objectName);
                }
            } catch (ObjectNotFoundException e) {
                throw new ServerClusterViewManagerException(objectName+" is missing !", e);
            }
        }
        return objectNamesInCluster;
    }

    /**
     * @return the set of name of object
     */
    public Set<String> getObjectNames() {
        Vector<String> objectNames = distributedTree.getChildrenNames(getObjectsPath());
        return new HashSet<String>(objectNames);
    }

    /**
     * Returns the time between each update of the cluster view by clients.
     * @return the time between each update of the cluster view by clients
     */
    public int getDelayToRefresh() {
        return (Integer) distributedTree.get(getDelayPath());
    }

    /**
     * Sets the time between each update of the cluster view by clients.
     * @param delay the time between each update of the cluster view by clients
     */
    public void setDelayToRefresh(final int delay) {
        distributedTree.reset(getDelayPath(), delay);
    }

    /**
     * Returns true if the pool for object with the given name should be empty.
     * @param objectName a name of object
     * @return true if the pool for object with the given name should be empty
     */
    public boolean isPoolToEmpty(final String objectName) {
        return distributedTree.exists(getPoolToEmptyPath(objectName));
    }

    /**
     * Adds the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     */
    public void addPoolToEmpty(final String objectName) {
        distributedTree.add(getPoolToEmptyPath(objectName));
    }

    /**
     * Removes the pool of the object with the given name of the list of pool that should be empty.
     * @param objectName a name of object
     */
    public void removePoolToEmpty(final String objectName) {
        distributedTree.remove(getPoolToEmptyPath(objectName));
    }

    /**
     * Return true if the server with the given reference if blacklisted.
     * @param serverRef a reference on a server
     * @return true the server with the given reference if blacklisted
     */
    public boolean isServerBlackListed(final ServerRef serverRef) {
        return distributedTree.exists(getServerBlackListedPath(
                serverRef.getInetAddress().getHostAddress()+"_"+serverRef.getPort()));
    }

    /**
     * Add a server to the blacklist.
     * @param serverRef a reference on a server
     */
    public void addServerToBlackList(final ServerRef serverRef) {
        distributedTree.add(getServerBlackListedPath(serverRef.getInetAddress().getHostAddress()+"_"+serverRef.getPort()));
    }

    /**
     * Remove a server to the blacklist.
     * @param serverRef a reference on a server
     */
    public void removeServerFromBlackList(final ServerRef serverRef) {
        distributedTree.remove(getServerBlackListedPath(serverRef.getInetAddress().getHostAddress()+"_"+serverRef.getPort()));
    }

    /**
     * Returns the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @return the load-factor for the server with the given address
     * @throws ServerNotFoundException if none server has the given address
     */
    public int getLoadFactor(final ServerRef serverRef) throws ServerNotFoundException {
        Object obj = distributedTree.get(
                getLoadFactorPath(serverRef.getInetAddress().getHostAddress()+"_"+serverRef.getPort()));
        if(obj == null) {
            throw new ServerNotFoundException("Unknown server: "+serverRef);
        }
        return (Integer) obj;
    }

    /**
     * Sets the load-factor for the server with the given address.
     * @param serverRef a reference on a server
     * @param loadFactor the load-factor for the server with the given address
     */
    public void setLoadFactor(final ServerRef serverRef, final int loadFactor) {
        distributedTree.reset(getLoadFactorPath(serverRef.getInetAddress().getHostAddress()+"_"+serverRef.getPort()), loadFactor);
    }

    /**
     * Initialize the statistics.
     */
    @Override
    protected void initStats() {
        synchronized (lock) {
            if(!distributedTree.exists(getNbClientsConnectedToProviderPath())) {
                distributedTree.add(getNbClientsConnectedToProviderPath(), 0);
            }
        }
    }

    /**
     * @return the numbers of clients connected to a provider of the cluster view
     */
    public int getNbClientsConnectedToProvider() {
        return (Integer) distributedTree.get(getNbClientsConnectedToProviderPath());
    }

    /**
     * Register a new client.
     * @param uuid the Universally Unique Identifier of the client
     */
    public void registerClient(final UUID uuid) {
        // Warning! Not synchronized at cluster-level
        synchronized (lock) {
            if(distributedTree.exists(getClientUUIDPath(uuid.toString()))) {
                return;
            }
            distributedTree.add(getClientUUIDPath(uuid.toString()));
            distributedTree.set(getNbClientsConnectedToProviderPath(),
                    (Integer) distributedTree.get(
                            getNbClientsConnectedToProviderPath())+1);
        }
    }


    /* ---------- Begin of definition for book-marks in the distributedTree -------------------- */

    /**
     * @param address
     * @param instRef
     * @return
     */
    private String getInstAddrPath(final String address, final String instRef) {
        return address+"/"+instRef;
    }

    /**
     * @param objectName a name of object
     * @return the path in the distributed tree for a given object
     */
    private String getObjectPath(final String objectName) {
        return getObjectsPath()+"/"+objectName;
    }

    /**
     * @return the path in the distributed tree for the root
     */
    private String getRootPath() {
        return "/root";
    }

    /**
     * @return the path in the distributed tree for the objects
     */
    private String getObjectsPath() {
        return getRootPath()+"/objects";
    }

    /**
     * @param objectPath a path in the distributed tree for a object
     * @param protocol a name of protocol
     * @return a path in the distributed tree for instances of a given object and protocol
     */
    private String getProtocolPath(final String objectPath, final String protocol) {
        return objectPath+"/"+protocol;
    }

    /**
     * @param protocolPath a path in the distributed tree for instances of a given object and protocol
     * @param objectName a name of instance
     * @return a path in the distributed tree for a instance
     */
    private String getCMIReferencePath(final String protocolPath, final String objectName) {
        return protocolPath+"/"+objectName;
    }

    /**
     * @param cmiReference a reference on a instance
     * @return a path in the distributed tree for a instance
     */
    private String getCMIReferencePath(final CMIReference cmiReference) {
        return getProtocolPath(getObjectPath(cmiReference.getObjectName()),
                cmiReference.getServerRef().getProtocol())+"/"
                +cmiReference.getServerRef().getInetAddress().getHostAddress()
                +"_"+cmiReference.getServerRef().getPort();
    }

    /**
     * @return the path in the distributed tree for the addresses
     */
    private String getAddressesPath() {
        return getRootPath()+"/addresses";
    }

    /**
     * @param hostname a name of host
     * @param port a number of port
     * @return a path in the distributed tree for a given host and port
     */
    private String getAddressPath(final String hostname, final int port) {
        return getAddressesPath()+"/"+hostname+"/"+port;
    }

    /**
     * @return the path in the distributed tree for the delay
     */
    private String getDelayPath() {
        return getRootPath()+"/delay";
    }

    /**
     * @return the path in the distributed tree for the black listed servers
     */
    private String getBlackListPath() {
        return getRootPath()+"/black_list";
    }

    /**
     * @return the path in the distributed tree for the pool to empty
     */
    private String getPoolToEmptyPath() {
        return getRootPath()+"/pool_to_empty";
    }

    /**
     * @return the path in the distributed tree for the load factors
     */
    private String getLoadFactorsPath() {
        return getRootPath()+"/load_factors";
    }

    /**
     * @return the path in the distributed tree for the statistics
     */
    private String getStatsPath() {
        return getRootPath()+"/stats";
    }

    /**
     * @return the path in the distributed tree to get the number of connected clients
     */
    private String getNbClientsConnectedToProviderPath() {
        return getStatsPath()+"/nbClientsOnProvider";
    }

    /**
     * @return the path in the distributed tree to get the list of the client UUIDs
     */
    private String getClientUUIDPath() {
        return getStatsPath()+"/clientUUID";
    }

    /**
     * @return the path in the distributed tree to retrieve the given UUID
     */
    private String getClientUUIDPath(final String uuid) {
        return getClientUUIDPath()+"/"+uuid;
    }

    /**
     * @param objectName a name of object
     * @return
     */
    private String getPoolToEmptyPath(final String objectName) {
        return getPoolToEmptyPath()+"/"+objectName;
    }

    /**
     * @param serverName a provider URL
     * @return
     */
    private String getServerBlackListedPath(final String serverName) {
        return getBlackListPath()+"/"+serverName;
    }

    /**
     * @param address
     * @return
     */
    private String getLoadFactorPath(final String address) {
        return getLoadFactorsPath()+"/"+address;
    }



    /* --------------- End of definition for book-marks in the distributedTree -------------------------- */

    /* ------------------ DistributedTree.DistributedTreeListener interface ------------ */

    public void nodeAdded(final String fqn, final Serializable element) {
        LOGGER.debug("{0} has been added", fqn);
    }

    public void nodeModified(final String fqn, final Serializable oldElement, final Serializable newElement) {
        LOGGER.debug("{0} has been modified", fqn);
        LOGGER.debug("Old element: {0} - New element: {1}", oldElement, newElement);
        // Updates local data if needed
        if(newElement != null && newElement instanceof DistributedObjectInfo) {
            DistributedObjectInfo newDistributedObjectInfo = (DistributedObjectInfo) newElement;
            synchronized (newDistributedObjectInfo) {
                String objectName = newDistributedObjectInfo.getObjectName();

                // Checks if LB policy has changed
                LBPolicyData newLBPolicyData = newDistributedObjectInfo.getLBPolicyData();
                if(oldElement == null || !((DistributedObjectInfo) oldElement).getLBPolicyData().equals(newLBPolicyData)) {
                    try {
                        updateLBPolicy(objectName);
                    } catch (Exception e) {
                        LOGGER.error("Cannot update LBPolicy for object with name {0}", objectName, e);
                        throw new RuntimeException("Cannot update LBPolicy for object with name "+objectName, e);
                    }
                }
            }
        }
    }

    public void nodeRemoved(final String fqn) {
        LOGGER.debug("{0} has been removed", fqn);
    }

    /* ---------------- End of DistributedTree.DistributedTreeListener interface -------- */

    /* ------------------ DistributedTree.ViewListener interface ------------ */

    public void viewChange(final Vector<Address> newMbrs, final Vector<Address> oldMbrs) {

        LOGGER.debug("view has changed");
        // Remove the reference on the old objects
        for(Address oldMbr : oldMbrs) {
            IpAddress ipAddr = (IpAddress) oldMbr;
            InetAddress inetAddress = ipAddr.getIpAddress();
            String hostname = inetAddress.getHostAddress();
            int port = ipAddr.getPort();
            LOGGER.debug("Search instances on {0}:{1}", hostname, String.valueOf(port));
            String addrPath = getAddressPath(hostname, port);
            Vector<String> childrenNames = distributedTree.getChildrenNames(addrPath);
            for(String instAddrName : childrenNames) {
                String instAddrPath = getInstAddrPath(addrPath, instAddrName);
                String pathToDelete = (String) distributedTree.get(instAddrPath);
                LOGGER.debug("Marking {0} to delete", pathToDelete);
                PATHS_TO_DELETE.add(pathToDelete);
            }
            LOGGER.debug("Marking {0} to delete...", addrPath);
            synchronized (PATHS_TO_DELETE) {
                PATHS_TO_DELETE.add(addrPath);
                resumableRunnable.resumeExecution();
            }
        }
    }

    /* ---------------- End of DistributedTree.ViewListener interface -------- */

}
