/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:CMIInitialContextFactory.java 914 2007-05-25 16:48:16Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.jndi.context;

import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.spi.InitialContextFactory;

import org.ow2.carol.cmi.config.CMIConfig;
import org.ow2.carol.cmi.config.CMIConfigException;
import org.ow2.carol.cmi.config.CMIPropertyHelper;
import org.ow2.carol.cmi.config.JNDIConfig;
import org.ow2.carol.cmi.reference.ServerRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


/**
 * CMIInitialContext factory.<br>
 * The following properties must be set in the environment given in parameter:
 * <ul>
 * <li><code>"cmi.context.wrapped.protocol"</code><br>
 * Define the protocol of the wrapped context.
 * For example: jrmp, iiop or irmi.
 * <li><code>"cmi.context.provider.urls"</code><br>
 * Define the initial set of provider URLs (it will be updated periodically).
 * For example: rmi://212.13.45.9:1099,212.13.45.9:1103,129.14.13.5:1099.
 * Supported schemes are rmi and iiop.<br>
 * The specified protocol must be consistent with the scheme of url,<br>
 * ie: jrmp as protocol with rmi as scheme is good but not jrmp with iiop !
 * <li><code>"cmi.context.wrapped.factory"</code><br>
 * Define the InitialContextFactory to use to construct the real context.
 * <li><code>"cmi.mode.server"</code><br>
 * Indicate if the server mode is enabled. The server mode allows performing bindings but need a local registry.
 * </ul>
 * @author The new CMI team
 * @see CMIContext
 */
public final class CMIInitialContextFactory implements InitialContextFactory {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(CMIInitialContextFactory.class);

    /**
     * Protocol of the CMIContext.
     */
    private String protocol;

    /**
     * Create a CMI context.<br>
     * Because the JNDI environment is provided in parameter,
     * if the expected properties are not set, a NamingException will be thrown.
     * @param environment The needed properties to create the CMI context.
     * @return A CMI context
     * @throws NamingException If cannot create an initial context.
     */
    public Context getInitialContext(final Hashtable<?, ?> environment) throws NamingException {

        boolean isServer = isServerMode(environment);

        if(isServer && !CMIConfig.isConfigured()) {
            Properties cmiProps = extractCMIConfig(environment);
            // Configure CMI
            try {
                if(cmiProps != null) {
                    CMIConfig.setProperties(cmiProps);
                } else {
                    // Read cmi.properties
                    CMIConfig.init();
                }
            } catch (CMIConfigException e) {
                logger.error("Cannot initialize CMI");
                NamingException ne = new NamingException("Cannot initialize CMI");
                ne.setRootCause(e);
                throw ne;
            }
        }

        // Configure JNDI with the given environment
        protocol = extractProtocol(environment);
        String wrappedFactoryName = extractWrappedFactoryName(environment);
        List<ServerRef> initialProviderURLs = extractServerRefs(environment);

        if(!isServer) {
            logger.debug("Context client is returned");
            return new CMIContext(initialProviderURLs, protocol, wrappedFactoryName);
        }

        int size = initialProviderURLs.size();
        if(size == 0) {
            logger.error("No providerURL found");
            throw new NamingException("No providerURL found");
        } else if(size > 1) {
            logger.warn("Only one providerURL is expected: the first is used.");
        }

        boolean replicationEnabled = isReplicationEnabled(environment);
        CMIContext cmiContext = new CMIContext(
                initialProviderURLs.get(0), wrappedFactoryName, replicationEnabled);
        if(replicationEnabled) {
            try {
                cmiContext.register();
            } catch (CMIContextException e) {
                logger.error("Cannot register the context");
                NamingException ne = new NamingException("Cannot register the context");
                ne.setRootCause(e);
                throw ne;
            }
        }
        return cmiContext;
    }


    /********* Extraction of the properties specific at CMI: *********/


    /**
     * @param environment an environment
     * @return properties for CMI
     */
    private Properties extractCMIConfig(final Hashtable<?, ?> environment) {

        Properties cmiProps = new Properties();
        boolean containCMIConfig = false;
        for(Object obj : environment.keySet()) {
            if(obj instanceof String) {
                String name = (String) obj;
                Object value =  environment.get(obj);
                if(value instanceof String) {
                    cmiProps.setProperty(name, (String) value);
                    if(CMIPropertyHelper.containProperty(name)) {
                        containCMIConfig = true;
                    }
                }
            }
        }
        if(!containCMIConfig) {
            logger.debug("No CMI property found");
            return null;
        } else {
            logger.debug("CMI properties found: "+cmiProps);
            return cmiProps;
        }
    }

    /**
     * Returns protocol from environment.
     * @param environment an environment
     * @return the protocol found in environment
     * @throws NamingException if environment doesn't contain protocol
     */
    private String extractProtocol(final Hashtable<?, ?> environment) throws NamingException {
        Object obj = environment.get(JNDIConfig.WRAPPED_PROTOCOL);
        if(obj == null) {
            logger.error("Property {0} is missing.", JNDIConfig.WRAPPED_PROTOCOL);
            throw new NamingException("Property "+JNDIConfig.WRAPPED_PROTOCOL+" is missing.");
        }
        return (String) obj;
    }

    /**
     * Returns the name of wrapped factory by extracting it from environment.
     * @param environment an environment
     * @return the name of wrapped factory found in environment
     * @throws NamingException if environment doesn't contain wrapped factory name
     */
    private String extractWrappedFactoryName(final Hashtable<?, ?> environment) throws NamingException {
        Object obj = environment.get(JNDIConfig.WRAPPED_INITIAL_CONTEXT_FACTORY);
        if(obj == null) {
            logger.error("Property {0} is missing.", JNDIConfig.WRAPPED_INITIAL_CONTEXT_FACTORY);
            throw new NamingException("Property "+JNDIConfig.WRAPPED_INITIAL_CONTEXT_FACTORY+" is missing.");
        }
        return (String) obj;
    }

    /**
     * Returns the initial provider URLs by extracting it from environment.
     * @param environment an environment
     * @return a list of references on server found in environment
     * @throws NamingException if environment doesn't contain provider URL or if the property is malformed
     */
    @SuppressWarnings("unchecked")
    private List<ServerRef> extractServerRefs(final Hashtable<?, ?> environment) throws NamingException {
        Object obj = environment.get(JNDIConfig.INITIAL_PROVIDER_URLS);
        if(obj == null) {
            logger.error("Property {0} is missing.", JNDIConfig.INITIAL_PROVIDER_URLS);
            throw new NamingException("Property "+JNDIConfig.INITIAL_PROVIDER_URLS+" is missing.");
        }
        List<ServerRef> serverRefs = new ArrayList<ServerRef>();

        for(String providerURL : (List<String>) obj) {
            try {
                serverRefs.add(new ServerRef(protocol, providerURL));
            } catch (MalformedURLException e) {
                logger.error("The following provider URL is malformed {0}", providerURL);
                NamingException ne = new NamingException("The following provider URL is malformed "+providerURL);
                ne.setRootCause(e);
                throw ne;
            } catch (UnknownHostException e) {
                logger.error("The host name of the provider URL {0} cannot be resolved", providerURL);
                NamingException ne = new NamingException("The host name of the provider URL "+providerURL+" cannot be resolved");
                ne.setRootCause(e);
                throw ne;
            }
        }
        return serverRefs;
    }

    /**
     * Returns true if the server mode must be used.
     * @param environment an environment
     * @return true if the server mode must be used
     */
    private boolean isServerMode(final Hashtable<?, ?> environment) {
        Object obj = environment.get(JNDIConfig.SERVER_MODE);
        if(obj == null) {
            logger.warn("Property {0} is not defined: use the default value 'false'", JNDIConfig.SERVER_MODE);
            return false;
        }
        return (Boolean) obj;
    }

    /**
     * Returns true if a manager of the cluster view must be used.
     * @param environment an environment
     * @return true if a manager of the cluster view must be used
     */
    private boolean isReplicationEnabled(final Hashtable<?, ?> environment) {
        Object obj = environment.get(JNDIConfig.REPLICATION_ENABLED);
        if(obj == null) {
            logger.warn("Property {0} is not defined: use the default value 'true'", JNDIConfig.REPLICATION_ENABLED);
            return true;
        }
        return (Boolean) obj;
    }

}
