/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: HASingletonPolicy.java 1336 2007-10-22 16:38:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.policy;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.lb.NoLoadBalanceableException;
import org.ow2.carol.cmi.lb.decision.BasicDecisionManager;
import org.ow2.carol.cmi.lb.decision.DecisionManager;
import org.ow2.carol.cmi.lb.decision.DecisionUtil;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of a policy of load-balancing that always favors one server in the cluster.
 * @author The new CMI team
 */
@ThreadSafe
public class HASingletonPolicy implements ILBPolicy<CMIReference> {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = -3247289458693993608L;

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(HASingletonPolicy.class);

    /**
     * The list of servers that contains the order of selection (index 0 is the first chosen).
     */
    private volatile List<String> serverRefs = new ArrayList<String>();

    /**
     * A policy to use if no singleton is declared.
     */
    private volatile ILBPolicy<CMIReference> lbPolicy;

    private volatile ILBStrategy<CMIReference> lbStrategy;

    /**
     * Construct a new instance of policy ha-singleton with the policy first available as rescue.
     */
    @SuppressWarnings("unchecked")
    public HASingletonPolicy() {
        try {
            lbPolicy = FirstAvailablePolicy.class.newInstance();
        } catch (Exception e) {
            LOGGER.debug("Cannot instanciate FirstAvailable.class", e);
        }
    }

    /**
     * Always choose the same server (called singleton) on any client.
     * @param cmiReferences a list of references
     * @throws NoLoadBalanceableException if no server available
     * @return the singleton
     */
    public synchronized CMIReference choose(final List<CMIReference> cmiReferences)
    throws NoLoadBalanceableException {
        if (cmiReferences.size() == 0) {
            LOGGER.error("The given list is empty");
            throw new NoLoadBalanceableException("The given list is empty");
        }

        // Search for an available singleton
        for(String serverRef : serverRefs) {
            for(CMIReference cmiReference : cmiReferences) {
                if(serverRef.equals(cmiReference.getServerRef().getProviderURL())) {
                    LOGGER.debug("Found a singleton: {0}", serverRef);
                    return cmiReference;
                }
            }
        }

        // None singleton is available, use an other policy.
        return lbPolicy.choose(cmiReferences);
    }

    /**
     * Returns a decision when an exception is thrown during an invocation for a given reference.
     * Also, remove the current master.
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param cmiReference the reference that has caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an invocation for a given load-balanceable
     */
    public DecisionManager<Void> onInvokeException(final Method method,
            final Object[] parameters, final CMIReference cmiReference, final Throwable thr) {
        if (DecisionUtil.mustFailoverOnInvoke(thr)) {
            serverRefs.remove(cmiReference.getServerRef().getProviderURL());
            return BasicDecisionManager.doRetry();
        }
        return BasicDecisionManager.doThrow();
    }

    /**
     * Returns a decision when an exception is thrown during an access to a registry for a given reference.
     * Also, remove the current master.
     * @param cmiReference the reference that has caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an access to a registry for a given reference
     */
    public DecisionManager<Void> onLookupException(final CMIReference cmiReference,
            final Throwable thr) {
        if (DecisionUtil.mustFailoverOnLookup(thr)) {
            serverRefs.remove(cmiReference.getServerRef().getProviderURL());
            return BasicDecisionManager.doRetry();
        }
        return BasicDecisionManager.doThrow();
    }

    /**
     * Returns a decision when the invocation of a remote method ends.
     * @param <ReturnType> the type of the returned value
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param cmiReference the reference used for the invocation
     * @param retVal the returned value
     * @return the decision when the invocation of a remote method ends
     */
    public <ReturnType> DecisionManager<ReturnType> onReturn(final Method method, final Object[] parameters,
            final CMIReference cmiReference, final ReturnType retVal) {
        LOGGER.debug("onReturn: do nothing !");
        return BasicDecisionManager.doReturn(retVal);
    }

    /**
     * Sets a strategy to modify the behavior of this policy.
     * It will be used only if no singleton is declared.
     * @param lbStrategy a strategy of load-balancing
     */
    public void setStrategy(final ILBStrategy<CMIReference> lbStrategy) {
        this.lbStrategy = lbStrategy;
        lbPolicy.setStrategy(lbStrategy);
    }

    public ILBStrategy<CMIReference> getStrategy() {
        return lbStrategy;
    }

    @SuppressWarnings("unchecked")
    public void setPolicyClassname(final String lbPolicyClassname) throws NoLoadBalanceableException {
        try {
            lbPolicy =
                (ILBPolicy<CMIReference>) Class.forName(lbPolicyClassname).newInstance();
        } catch (Exception e) {
            throw new NoLoadBalanceableException("Cannot instanciate the rescue policy", e);
        }
        lbPolicy.setStrategy(lbStrategy);
    }

    public String getPolicyClassname() throws NoLoadBalanceableException {
        return lbPolicy.getClass().getName();
    }

    /**
     * Return the order to elect a singleton in the cluster.
     * The first element will be the first elected.
     * @return a list of reference on servers (e.g. {rmi://localhost:9000})
     */
    public List<String> getSingletons() {
        return serverRefs;
    }

    /**
     * Set the order to elect a singleton in the cluster.
     * The first element will be the first elected.
     * @param serverRefs a list of reference on servers (e.g. {rmi://localhost:9000})
     */
    public synchronized void setSingletons(final List<String> serverRefs) {
        this.serverRefs=serverRefs;
    }

    /**
     * Add a server in the list of singleton at the first position (it will be the new master).
     * @param serverRef a reference on server (e.g. rmi://localhost:9000)
     */
    public synchronized void setSingleton(final String serverRef) {
        serverRefs.add(0, serverRef);
    }

    public synchronized String getSingleton() {
        return serverRefs.get(0);
    }

    @Override
    public String toString() {
        return "HASingletonPolicy - ServerRefs: "+serverRefs
        +" - master: "+serverRefs.get(0)+" - Rescue policy: "+lbPolicy;
    }

}
