/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:LBPolicyFactory.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.util;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.lb.PropertyConfigurationException;
import org.ow2.carol.cmi.lb.data.LBPropertyData;
import org.ow2.carol.cmi.lb.policy.ILBPolicy;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Define a factory to construct policies.
 * @param <T> the type parameter of the constructed policies
 * @author The new CMI team
 */
@ThreadSafe
public final class LBPolicyFactory<T extends LoadBalanceable> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(LBPolicyFactory.class);

    /**
     * A manager to retrieve the cluster view.
     */
    private final ClusterViewManager clusterViewManager;

    /**
     *
     */
    private static ConcurrentHashMap<String, ConcurrentHashMap<String, LBPropertyData>> propertyData =
        new ConcurrentHashMap<String, ConcurrentHashMap<String, LBPropertyData>>();

    /**
     * Constructs a new factory for policies with a given manager of cluster view.
     * @param clusterViewManager a manager to retrieve the cluster view
     */
    public LBPolicyFactory(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
    }


    /**
     * Returns a policy to access to the object with the given name.
     * @param objectName a name of object
     * @return a policy to access to the object with the given name
     * @throws LBPolicyFactoryException if the policy cannot be constructed
     * @throws ObjectNotFoundException if none object has the given name
     */
    public ILBPolicy<T>  getLBPolicy(final String objectName) throws LBPolicyFactoryException, ObjectNotFoundException {
        ILBPolicy<T> lbPolicy = createLBPolicy(objectName);
        try {
            findProperties(lbPolicy.getClass());
        } catch (PropertyConfigurationException e) {
            LOGGER.error("Error while finding properties", e);
            throw new LBPolicyFactoryException("Error while finding properties", e);
        }
        Map<String, Object>  properties = clusterViewManager.getPropertiesForLBPolicy(objectName);
        if(properties != null) {
            try {
                setProperties(lbPolicy, properties);
            } catch (PropertyConfigurationException e) {
                LOGGER.error("Error while configuring the properties", e);
                throw new LBPolicyFactoryException("Error while configuring the properties", e);
            }
        }
        ILBStrategy<T> lbStrategy = createLBStrategy(objectName);
        lbPolicy.setStrategy(lbStrategy);
        return lbPolicy;
    }

    /**
     * Returns a pure policy to access to the object with the given name.
     * @param objectName a name of object
     * @return a pure policy to access to the object with the given name
     * @throws LBPolicyFactoryException if the policy cannot be constructed
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    public ILBPolicy<T> createLBPolicy(final String objectName) throws LBPolicyFactoryException, ObjectNotFoundException {

        Class<? extends ILBPolicy> lbPolicyClass = clusterViewManager.getLBPolicyClass(objectName);

        ILBPolicy<T> lbPolicy;
        try {
            lbPolicy = (ILBPolicy<T>) lbPolicyClass.newInstance();
        } catch (Exception e) {
            LOGGER.error("Cannot construct the LB policy", e);
            throw new LBPolicyFactoryException("Cannot construct the LB policy", e);
        }
        return lbPolicy;
    }

    /**
     * Sets properties for a given policy.
     * @param lbPolicy a policy
     * @param properties properties of the policy
     * @throws PropertyConfigurationException if a property cannot be set
     */
    public static void setProperties(final ILBPolicy<?> lbPolicy, final Map<String, Object> properties)
    throws PropertyConfigurationException {
        for(String propertyName : properties.keySet()) {
            Object propertyValue = properties.get(propertyName);
            setProperty(lbPolicy, propertyName, propertyValue);
        }
    }

    /**
     * Set a property for a given policy.
     * @param lbPolicy a policy
     * @param propertyName a name of property
     * @param propertyValue a value of property
     * @throws PropertyConfigurationException
     */
    @SuppressWarnings("unchecked")
    public static void setProperty(
            final ILBPolicy<?> lbPolicy, final String propertyName, final Object propertyValue)
    throws PropertyConfigurationException {
        Class<? extends ILBPolicy> lbPolicyClass = lbPolicy.getClass();
        LBPropertyData lbPropertyData =
            propertyData.get(lbPolicyClass.getName()).get(propertyName);
        try {
            lbPropertyData.getSetter().invoke(lbPolicy, propertyValue);
        } catch (Exception e) {
            LOGGER.error("Cannot set the property for name {0} with {1}",
                    propertyName, propertyValue, e);
            throw new PropertyConfigurationException(
                    "Cannot set the property for name "
                    + propertyName + " with "
                    + propertyValue, e);
        }
    }

    @SuppressWarnings("unchecked")
    public static Map<String, Object> getProperties(final ILBPolicy<?> lbPolicy) {
        Class<? extends ILBPolicy> lbPolicyClass = lbPolicy.getClass();
        Map<String, LBPropertyData> lbPropertyData =
            propertyData.get(lbPolicyClass.getName());
        Map<String, Object> properties = new HashMap<String, Object>();
        Object propertyValue;
        for(String propertyName : lbPropertyData.keySet()) {
            propertyValue = getProperty(lbPolicy, propertyName);
            properties.put(propertyName, propertyValue);
        }
        return properties;
    }

    @SuppressWarnings("unchecked")
    public static Object getProperty(final ILBPolicy<?> lbPolicy, final String propertyName) {
        Class<? extends ILBPolicy> lbPolicyClass = lbPolicy.getClass();
        Map<String, LBPropertyData> lbPropertyData =
            propertyData.get(lbPolicyClass.getName());
        LBPropertyData property = lbPropertyData.get(propertyName);
        try {
            return property.getGetter().invoke(lbPolicy);
        } catch (Exception e) {
            LOGGER.error("Cannot invoke the getter for property {0}", propertyName, e);
            return null;
        }
    }

    /**
     * Returns a policy to access to the object with the given name.
     * @param objectName a name of object
     * @return a strategy to access to the object with the given name
     * @throws LBPolicyFactoryException if the strategy cannot be constructed
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    private ILBStrategy<T> createLBStrategy(final String objectName)
    throws LBPolicyFactoryException, ObjectNotFoundException {

        Class<? extends ILBStrategy> lbStrategyClass = clusterViewManager.getLBStrategyClass(objectName);

        ILBStrategy<T> lbStrategy = null;

        // Lookup if it exists a constructor with a parameter of type ClusterviewManager
        Constructor<?>[] constructors = lbStrategyClass.getDeclaredConstructors();
        for(Constructor<?> constructor : constructors) {
            Class<?>[] paramTypes = constructor.getParameterTypes();
            if(paramTypes.length == 1 && paramTypes[0].equals(ClusterViewManager.class)) {
                try {
                    lbStrategy = (ILBStrategy<T>) constructor.newInstance(clusterViewManager);
                } catch (Exception e) {
                    LOGGER.error("Cannot construct the LB strategy with the instance of ClusterViewManager", e);
                    throw new LBPolicyFactoryException(
                            "Cannot construct the LB strategy with the instance of ClusterViewManager", e);
                }
                break;
            }
        }

        // Use the default constructor
        if(lbStrategy == null) {
            try {
                lbStrategy = (ILBStrategy<T>) lbStrategyClass.newInstance();
            } catch (Exception e) {
                LOGGER.error("Cannot construct the LB strategy with the default constructor", e);
                throw new LBPolicyFactoryException("Cannot construct the LB strategy with the default constructor", e);
            }
        }
        return lbStrategy;
    }

    public static Type getPropertyRawType(final String lbPolicyClassname, final String propertyName) {
        Type propertyType = getPropertyType(lbPolicyClassname, propertyName);
        if(propertyType instanceof ParameterizedType) {
            return ((ParameterizedType) propertyType).getRawType();
        } else {
            return propertyType;
        }
    }

    public static Type getPropertyType(final String lbPolicyClassname, final String propertyName) {
        return propertyData.get(lbPolicyClassname).get(propertyName).getLbPropertyType();
    }

    public static Map<String, LBPropertyData> getPropertyData(final String lbPolicyClassname) {
        return propertyData.get(lbPolicyClassname);
    }

    /**
     * Converts the given value from String to the given type.
     * @param lbPropertyName
     * @param svalue a value
     * @return an object that has the same that the given field
     * @throws PropertyConfigurationException if the conversion is not possible
     */
    @SuppressWarnings("unchecked")
    public static Object convertString(
            final Class<? extends ILBPolicy> lbPolicyClass,
            final String lbPropertyName,
            final String svalue) throws PropertyConfigurationException {
        Type propertyType = getPropertyType(lbPolicyClass.getName(), lbPropertyName);
        return convertString(propertyType, svalue);
    }

    public static Object convertString(
            final Type propertyType,
            final String svalue) throws PropertyConfigurationException {
        if(propertyType instanceof ParameterizedType) {
            throw new PropertyConfigurationException("");
        }
        Class<?> klass = (Class<?>) propertyType;
        Object propertyValue;
        //  Attempt to cast String forward a primitive type
        if(klass.equals(String.class)) {
            propertyValue = svalue;
        } else if(klass.equals(boolean.class)) {
            propertyValue = Boolean.valueOf(svalue);
        } else if(klass.equals(byte.class)) {
            propertyValue = Byte.valueOf(svalue);
        } else if(klass.equals(int.class)) {
            propertyValue = Integer.valueOf(svalue);
        } else if(klass.equals(long.class)) {
            propertyValue = Long.valueOf(svalue);
        } else if(klass.equals(float.class)) {
            propertyValue = Float.valueOf(svalue);
        } else if(klass.equals(double.class)) {
            propertyValue = Double.valueOf(svalue);
        } else if(klass.equals(URL.class)){ // Try with some complex type
            try {
                propertyValue = new URL(svalue);
            } catch (MalformedURLException e) {
                throw new PropertyConfigurationException("Cannot convert the string "+svalue+" to an URL", e);
            }
        } else if(klass.equals(File.class)) {
            propertyValue = new File(svalue);
        } else if(klass.equals(Class.class)) {
            try {
                propertyValue = Class.forName(svalue);
            } catch (ClassNotFoundException e) {
                throw new PropertyConfigurationException("Cannot convert the string "+svalue+" to a class", e);
            }
        } else {
            LOGGER.error("Type of field not supported: {0}", klass.getName());
            throw new PropertyConfigurationException(
                    "Type of field not supported: "+klass.getName());
        }
        return propertyValue;
    }

    @SuppressWarnings("unchecked")
    public static List<?> convertStrings(
            final Class<? extends ILBPolicy> lbPolicyClass,
            final String lbPropertyName,
            final List<String> slist)throws PropertyConfigurationException {
        Type parameterizedType = getPropertyType(lbPolicyClass.getName(), lbPropertyName);
        if(!(parameterizedType instanceof ParameterizedType)) {
            throw new PropertyConfigurationException("");
        }
        Type propertyType = ((ParameterizedType) parameterizedType).getActualTypeArguments()[0];
        List<Object> values = new ArrayList<Object>();
        for(String svalue : slist) {
            values.add(convertString(propertyType, svalue));
        }
        return values;
    }


    @SuppressWarnings("unchecked")
    public static void findProperties(final Class<? extends ILBPolicy> lbPolicyClass)
    throws PropertyConfigurationException {
        if(propertyData.containsKey(lbPolicyClass.getName())) {
            return;
        }
        ConcurrentHashMap<String, LBPropertyData> properties =
            new ConcurrentHashMap<String, LBPropertyData>();
        for(Method method : lbPolicyClass.getDeclaredMethods()) {
            if(method.getName().startsWith("get")) {
                LBPropertyData lbPropertyData =
                    new LBPropertyData(method, lbPolicyClass);
                properties.putIfAbsent(
                        lbPropertyData.getLbPropertyName(), lbPropertyData);
            }
        }
        propertyData.putIfAbsent(
                lbPolicyClass.getName(), properties);
    }

}
