/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:RandomPolicy.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.policy;

import java.lang.reflect.Method;
import java.util.List;
import java.util.Random;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.lb.NoLoadBalanceableException;
import org.ow2.carol.cmi.lb.decision.BasicDecisionManager;
import org.ow2.carol.cmi.lb.decision.DecisionUtil;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


/**
 * Implementation of a policy of load-balancing that always selects randomly a load-balanceable.
 * @param <T> The type of object that was load-balanced
 * @author The new CMI team
 */
@ThreadSafe
public final class RandomPolicy<T extends LoadBalanceable> implements ILBPolicy<T> {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = -1339141265656236119L;

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(RandomPolicy.class);

    /**
     * Random numbers.
     */
    private final Random random = new Random();

    /**
     * A strategy to modify the behavior of this policy.
     */
    private ILBStrategy<T> lbStrategy = null;

    /**
     * Chooses randomly a load-balanceable among the list of load-balanceables.
     * @param loadBalanceables the list of load-balanceables
     * @throws NoLoadBalanceableException if no server available
     * @return the chosen load-balanceable
     */
    public synchronized T choose(final List<T> loadBalanceables) throws NoLoadBalanceableException{
        if (loadBalanceables.size() == 0){
            LOGGER.error("The given list is empty");
            throw new NoLoadBalanceableException("The given list is empty");
        }
        List<T> cmiRefsWithStrategy;
        if(lbStrategy != null) {
            cmiRefsWithStrategy = lbStrategy.choose(loadBalanceables);
            // If no server corresponds at this strategy, we don't use it
            if(cmiRefsWithStrategy.isEmpty()) {
                cmiRefsWithStrategy = loadBalanceables;
            }
        } else {
            cmiRefsWithStrategy = loadBalanceables;
        }
        int index = random.nextInt(cmiRefsWithStrategy.size());
        return cmiRefsWithStrategy.get(index);
    }

    /**
     * Returns a decision when an exception is thrown during an invocation for a given load-balanceable.
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param loadBalanceable the load-balanceable that have caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an invocation for a given load-balanceable
     */
    public BasicDecisionManager<Void> onInvokeException(final Method method, final Object[] parameters,
            final T loadBalanceable, final Throwable thr) {
        if (DecisionUtil.mustFailoverOnInvoke(thr)) {
            // TODO something
            return BasicDecisionManager.doRetry();
        }
        return BasicDecisionManager.doThrow();
    }

    /**
     * Returns a decision when the invocation of a remote method ends.
     * @param <ReturnType> the type of the returned value
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param loadBalanceable the load-balanceable used for the invocation
     * @param retVal the returned value
     * @return the decision when the invocation of a remote method ends
     */
    public <ReturnType> BasicDecisionManager<ReturnType> onReturn(final Method method, final Object[] parameters,
            final T loadBalanceable, final ReturnType retVal) {
        LOGGER.debug("onReturn: do nothing !");
        return BasicDecisionManager.doReturn(retVal);
    }

    /**
     * Returns a decision when an exception is thrown during an access to a registry for a given load-balanceable.
     * @param loadBalanceable the load-balanceable that have caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an access to a registry for a given load-balanceable
     */
    public BasicDecisionManager<Void> onLookupException(final T loadBalanceable, final Throwable thr) {
        if (DecisionUtil.mustFailoverOnLookup(thr)) {
            // TODO something
            return BasicDecisionManager.doRetry();
        }
        return BasicDecisionManager.doThrow();
    }

    /**
     * Sets a strategy to modify the behavior of this policy.
     * @param lbStrategy a strategy of load-balancing
     */
    public synchronized void setStrategy(final ILBStrategy<T> lbStrategy) {
        this.lbStrategy = lbStrategy;
    }

    @Override
    public String toString() {
        return "RandomPolicy - LB strategy: "+lbStrategy;
    }
}
