/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:LoadFactor.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.strategy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import net.jcip.annotations.Immutable;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Defines a strategy that order the list of servers with the load-factor.
 * @author The new CMI team
 */
@Immutable
public class LoadFactor implements ILBStrategy<CMIReference> {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = -7859116721872110483L;

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(LoadFactor.class);

    /**
     * The manager of the cluster view.
     */
    private final ClusterViewManager clusterViewManager;

    /**
     * Constructs a strategy for load-factor.
     * @param clusterViewManager the manager of the cluster view
     */
    public LoadFactor(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
    }

    /**
     * Returns a list of CMIReference where servers, that have a minimal load factor,
     * have their reference at the begin of the list.
     * @param cmiRefs a list of CMIReference
     * @return a list of CMIReference ordered by load-factor
     */
    public List<CMIReference> choose(final List<CMIReference> cmiRefs) {
        List<CMIReference> orderedCMIRefs = new ArrayList<CMIReference>(cmiRefs);
        Collections.sort(orderedCMIRefs, new LoadFactorComparator());
        return orderedCMIRefs;
    }

    @Override
    public String toString() {
        return "LoadFactor";
    }

    /**
     * A comparator of load-factors.
     * @author The new CMI team
     */
    private final class LoadFactorComparator implements Comparator<CMIReference> {

        /**
         * Compares the load-factors of two CMIReferences with their {@link ServerRef}.
         * @param cmiRef1 the first CMIReference
         * @param cmiRef2 the second CMIReference
         * @return a negative integer, zero, or a positive integer as the first server has a smaller,
         * equal to, or greater load-factor than than the second
         */
        public int compare(final CMIReference cmiRef1, final CMIReference cmiRef2) {
            int factor1;
            int factor2;
            try {
                factor1 = clusterViewManager.getLoadFactor(cmiRef1.getServerRef());
                factor2 = clusterViewManager.getLoadFactor(cmiRef2.getServerRef());
            } catch (ServerNotFoundException e) {
                throw new RuntimeException(e);
            }
            return new Integer(factor1).compareTo(factor2);
        }

    }

}
