/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: AbsLBPolicy.java 1597 2008-01-28 13:15:47Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.policy;

import java.lang.reflect.Method;
import java.util.List;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.lb.NoLoadBalanceableException;
import org.ow2.carol.cmi.lb.decision.BasicDecisionManager;
import org.ow2.carol.cmi.lb.decision.DecisionUtil;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Abstract implementation of a {@link ILBPolicy}, using the class {@link DecisionUtil} to take a decision and the class {@link BasicDecisionManager} to execute the decision.
 * @author Loris Bouzonnet
 * @param <T> The type of object that was load-balanced
 * @see DecisionUtil
 * @see BasicDecisionManager
 */
public abstract class AbsLBPolicy<T extends LoadBalanceable> implements ILBPolicy<T> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(AbsLBPolicy.class);

    /**
     * The manager of the cluster view.
     */
    private ClusterViewManager clusterViewManager;

    /**
     * A strategy to modify the behavior of this policy.
     */
    private ILBStrategy<T> lbStrategy = null;

    /**
     * Default constructor.
     */
    public AbsLBPolicy() {
        clusterViewManager = null;
    }

    /**
     * @param clusterViewManager the manager of the cluster view
     */
    public AbsLBPolicy(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
    }

    /**
     * Chooses a load-balanceable among the list of load-balanceables.
     * @param loadBalanceables a list of load-balanceables
     * @throws NoLoadBalanceableException if no server is available
     * @return the chosen load-balanceable
     */
    public abstract T choose(List<T> loadBalanceables) throws NoLoadBalanceableException;

    /**
     * Returns a decision when an exception is thrown during an invocation for a given load-balanceable.
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param loadBalanceable the load-balanceable that have caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an invocation for a given load-balanceable
     */
    public BasicDecisionManager<Void> onInvokeException(final Method method, final Object[] parameters,
            final T loadBalanceable, final Throwable thr) {
        if (DecisionUtil.mustFailoverOnInvoke(thr, clusterViewManager, loadBalanceable)) {
            return BasicDecisionManager.doRetry();
        }
        return BasicDecisionManager.doThrow();
    }

    /**
     * Returns a decision when the invocation of a remote method ends.
     * @param <ReturnType> the type of the returned value
     * @param method the method that was invoked
     * @param parameters the parameters of the method
     * @param loadBalanceable the load-balanceable used for the invocation
     * @param retVal the returned value
     * @return the decision when the invocation of a remote method ends
     */
    public <ReturnType> BasicDecisionManager<ReturnType> onReturn(final Method method, final Object[] parameters,
            final T loadBalanceable, final ReturnType retVal) {
        LOGGER.debug("onReturn: do nothing !");
        return BasicDecisionManager.doReturn(retVal);
    }

    /**
     * Returns a decision when an exception is thrown during an access to a registry for a given load-balanceable.
     * @param loadBalanceable the load-balanceable that have caused the exception
     * @param thr the exception that is thrown
     * @return the decision when an exception is thrown during an access to a registry for a given load-balanceable
     */
    public BasicDecisionManager<Void> onLookupException(final T loadBalanceable, final Throwable thr) {
        if (DecisionUtil.mustFailoverOnLookup(thr, clusterViewManager, loadBalanceable)) {
            return BasicDecisionManager.doRetry();
        }
        return BasicDecisionManager.doThrow();
    }

    /**
     * Sets a strategy to modify the behavior of this policy.
     * @param lbStrategy a strategy of load-balancing
     */
    public synchronized void setLBStrategy(final ILBStrategy<T> lbStrategy) {
        this.lbStrategy = lbStrategy;
    }

    /**
     * Return a strategy to modify the behavior of this policy.
     * @return a strategy to modify the behavior of this policy
     */
    public ILBStrategy<T> getLBStrategy() {
        return lbStrategy;
    }

    /**
     * Set the manager of the cluster view.
     * @param clusterViewManager the manager of the cluster view
     */
    public void setClusterViewManager(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
    }

}
