/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: HASingletonPolicy.java 1597 2008-01-28 13:15:47Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.policy;

import java.util.ArrayList;
import java.util.List;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.lb.NoLoadBalanceableException;

import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of a policy of load-balancing that always favors one server in the cluster.
 * @author The new CMI team
 */
@ThreadSafe
public class HASingletonPolicy extends AbsLBPolicy<CMIReference> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(HASingletonPolicy.class);

    /**
     * The list of servers that contains the order of selection (index 0 is the first chosen).
     */
    private volatile List<String> serverRefs = new ArrayList<String>();

    /**
     * A policy to use if no singleton is declared.
     */
    private volatile ILBPolicy<CMIReference> lbPolicy;

    /**
     * Construct a new instance of policy ha-singleton with the policy first available as rescue.
     */
    @SuppressWarnings("unchecked")
    public HASingletonPolicy() {
        try {
            lbPolicy = FirstAvailablePolicy.class.newInstance();
        } catch (Exception e) {
            LOGGER.debug("Cannot instanciate FirstAvailable.class", e);
        }
    }

    /**
     * Always choose the same server (called singleton) on any client.
     * @param cmiReferences a list of references
     * @throws NoLoadBalanceableException if no server available
     * @return the singleton
     */
    public synchronized CMIReference choose(final List<CMIReference> cmiReferences)
    throws NoLoadBalanceableException {
        if (cmiReferences.size() == 0) {
            LOGGER.error("The given list is empty");
            throw new NoLoadBalanceableException("The given list is empty");
        }

        // Search for an available singleton
        for(String serverRef : serverRefs) {
            for(CMIReference cmiReference : cmiReferences) {
                if(serverRef.equals(cmiReference.getServerRef().getProviderURL())) {
                    LOGGER.debug("Found a singleton: {0}", serverRef);
                    return cmiReference;
                }
            }
        }

        // None singleton is available, use an other policy.
        return lbPolicy.choose(cmiReferences);
    }

    /**
     * Sets a strategy to modify the behavior of the rescue policy.
     * It will be used only if no singleton is declared.
     * @param lbStrategy a strategy of load-balancing
     */
    @Override
    public void setLBStrategy(final ILBStrategy<CMIReference> lbStrategy) {
        super.setLBStrategy(lbStrategy);
        lbPolicy.setLBStrategy(lbStrategy);
    }

    /**
     * Return the order to elect a singleton in the cluster.
     * The first element will be the first elected.
     * @return a list of reference on servers (e.g. {rmi://localhost:9000})
     */
    public List<String> getSingletons() {
        return serverRefs;
    }

    /**
     * Set the order to elect a singleton in the cluster.
     * The first element will be the first elected.
     * @param serverRefs a list of reference on servers (e.g. {rmi://localhost:9000})
     */
    public synchronized void setSingletons(final List<String> serverRefs) {
        this.serverRefs=serverRefs;
    }

    /**
     * Add a server in the list of singleton at the first position (it will be the new master).
     * @param serverRef a reference on server (e.g. rmi://localhost:9000)
     */
    public synchronized void setSingleton(final String serverRef) {
        serverRefs.add(0, serverRef);
    }

    /**
     * @return the reference on server that is at the first position
     */
    public synchronized String getSingleton() {
        return serverRefs.get(0);
    }

    @Override
    public String toString() {
        return "HASingletonPolicy[ServerRefs: " + serverRefs
        + " - master: " + serverRefs.get(0)
        + " - Rescue policy: " + lbPolicy
        + " - Rescue strategy: " + getLBStrategy() + "]";
    }

}
