/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:RandomPolicy.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.policy;

import java.util.List;
import java.util.Random;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.lb.NoLoadBalanceableException;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


/**
 * Implementation of a policy of load-balancing that always selects randomly a load-balanceable.
 * @param <T> The type of object that was load-balanced
 * @author The new CMI team
 */
@ThreadSafe
public final class RandomPolicy<T extends LoadBalanceable> extends AbsLBPolicy<T> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(RandomPolicy.class);

    /**
     * Random numbers.
     */
    private final Random random = new Random();

    /**
     * Build the random policy.
     */
    public RandomPolicy() {
    }

    /**
     * Chooses randomly a load-balanceable among the list of load-balanceables.
     * @param loadBalanceables the list of load-balanceables
     * @throws NoLoadBalanceableException if no server available
     * @return the chosen load-balanceable
     */
    public synchronized T choose(final List<T> loadBalanceables) throws NoLoadBalanceableException{
        if (loadBalanceables.size() == 0){
            LOGGER.error("The given list is empty");
            throw new NoLoadBalanceableException("The given list is empty");
        }
        List<T> cmiRefsWithStrategy;

        ILBStrategy<T> lbStrategy = getLBStrategy();

        if(lbStrategy != null) {
            cmiRefsWithStrategy = lbStrategy.choose(loadBalanceables);
            // If no server corresponds at this strategy, we don't use it
            if(cmiRefsWithStrategy.isEmpty()) {
                cmiRefsWithStrategy = loadBalanceables;
            }
        } else {
            cmiRefsWithStrategy = loadBalanceables;
        }
        int index = random.nextInt(cmiRefsWithStrategy.size());
        return cmiRefsWithStrategy.get(index);
    }

    @Override
    public String toString() {
        return "RandomPolicy[LB strategy: " + getLBStrategy() + "]";
    }
}
