/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:LoadFactor.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.strategy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import net.jcip.annotations.Immutable;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Defines a strategy that order the list of servers with the load-factor.
 * @author The new CMI team
 */
@Immutable
public class LoadFactorSort implements ILBStrategy<CMIReference> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(LoadFactorSort.class);

    /**
     * The comparator of load-factors to sort the list.
     */
    private final LoadFactorComparator comparator;

    /**
     * Constructs a strategy for load-factor.
     * @param clusterViewManager the manager of the cluster view
     */
    public LoadFactorSort(final ClusterViewManager clusterViewManager) {
        comparator = new LoadFactorComparator(clusterViewManager);
    }

    /**
     * Returns a list of CMIReference where servers, that have a minimal load factor,
     * have their reference at the begin of the list.
     * @param cmiRefs a list of CMIReference
     * @return a list of CMIReference ordered by load-factor
     */
    public List<CMIReference> choose(final List<CMIReference> cmiRefs) {
        List<CMIReference> orderedCMIRefs = new ArrayList<CMIReference>(cmiRefs);
        Collections.sort(orderedCMIRefs, comparator);
        return orderedCMIRefs;
    }

    @Override
    public String toString() {
        return "LoadFactor";
    }

    /**
     * A comparator of load-factors.
     * @author The new CMI team
     */
    private class LoadFactorComparator implements Comparator<CMIReference> {

        /**
         * The manager of the cluster view.
         */
        private final ClusterViewManager clusterViewManager;

        /**
         * Construct a comparator of load-factor.
         * The load-factors are provided by the {@link ClusterViewManager}.
         * @param clusterViewManager the manager of the cluster view
         */
        private LoadFactorComparator(final ClusterViewManager clusterViewManager) {
            this.clusterViewManager = clusterViewManager;
        }

        /**
         * Compares the load-factors of two CMIReferences with their {@link ServerRef}.
         * @param cmiRef1 the first CMIReference
         * @param cmiRef2 the second CMIReference
         * @return a negative integer, zero, or a positive integer as the first server has a smaller,
         * equal to, or greater load-factor than than the second
         */
        public int compare(final CMIReference cmiRef1, final CMIReference cmiRef2) {
            ServerRef serverRef1 = cmiRef1.getServerRef();
            ServerRef serverRef2 = cmiRef2.getServerRef();
            try {
                int factor1 = clusterViewManager.getLoadFactor(serverRef1);
                int factor2 = clusterViewManager.getLoadFactor(serverRef2);
                return new Integer(factor1).compareTo(factor2);
            } catch (ServerNotFoundException e) {
                LOGGER.error("Cannot compare the load factors of servers {0} and {1}.", serverRef1, serverRef2, e);
                return 0;
            }
        }

    }

}
