/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:LBPolicyFactory.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.util;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.lb.PropertyConfigurationException;
import org.ow2.carol.cmi.lb.data.LBPropertyData;
import org.ow2.carol.cmi.lb.policy.ILBPolicy;
import org.ow2.carol.cmi.lb.strategy.ILBStrategy;
import org.ow2.carol.cmi.reference.ObjectNotFoundException;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Define a factory to construct policies.
 * @param <T> the type parameter of the constructed policies
 * @author The new CMI team
 */
@ThreadSafe
public final class LBPolicyFactory<T extends LoadBalanceable> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(LBPolicyFactory.class);

    /**
     * A manager to retrieve the cluster view.
     */
    private final ClusterViewManager clusterViewManager;

    /**
     * Map each object name with its properties of load-balancing.
     */
    private static ConcurrentHashMap<String, ConcurrentHashMap<String, LBPropertyData>> propertyData =
        new ConcurrentHashMap<String, ConcurrentHashMap<String, LBPropertyData>>();

    /**
     * Constructs a new factory for policies with a given manager of cluster view.
     * @param clusterViewManager a manager to retrieve the cluster view
     */
    public LBPolicyFactory(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
    }


    /**
     * Returns a policy to access to the object with the given name.
     * @param objectName a name of object
     * @return a policy to access to the object with the given name
     * @throws LBPolicyFactoryException if the policy cannot be constructed
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    public ILBPolicy<T>  getLBPolicy(final String objectName)
    throws LBPolicyFactoryException, ObjectNotFoundException {
        Class<? extends ILBPolicy<T>> lbPolicyClass =
            (Class<? extends ILBPolicy<T>>) clusterViewManager.getLBPolicyClass(objectName);
        Map<String, Object>  properties = clusterViewManager.getPropertiesForLBPolicy(objectName);
        Class<? extends ILBStrategy<T>> lbStrategyClass =
            (Class<? extends ILBStrategy<T>>) clusterViewManager.getLBStrategyClass(objectName);
        return getLBPolicy(lbPolicyClass, lbStrategyClass, properties);
    }

    /**
     * Returns a policy for the given classes of LB policy, strategy and properties.
     * @param objectName a name of object
     * @return a policy to access to the object with the given name
     * @throws LBPolicyFactoryException if the policy cannot be constructed
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    public ILBPolicy<T>  getLBPolicy(
            final Class<? extends ILBPolicy> lbPolicyClass, final Class<? extends ILBStrategy> lbStrategyClass,
            final Map<String, ?>  properties)
    throws LBPolicyFactoryException {
        ILBPolicy<T> lbPolicy = createLBPolicy(lbPolicyClass);
        try {
            findProperties(lbPolicy.getClass());
        } catch (PropertyConfigurationException e) {
            LOGGER.error("Error while finding properties", e);
            throw new LBPolicyFactoryException("Error while finding properties", e);
        }
        if(properties != null) {
            try {
                setProperties(lbPolicy, properties);
            } catch (PropertyConfigurationException e) {
                LOGGER.error("Error while configuring the properties", e);
                throw new LBPolicyFactoryException("Error while configuring the properties", e);
            }
        }
        ILBStrategy<T> lbStrategy = createLBStrategy(lbStrategyClass);
        lbPolicy.setLBStrategy(lbStrategy);
        return lbPolicy;
    }

    /**
     * Find properties in a class defining LB policy.
     * @param lbPolicyClass a class defining LB policy
     * @throws PropertyConfigurationException
     */
    @SuppressWarnings("unchecked")
    private static void findProperties(final Class<? extends ILBPolicy> lbPolicyClass)
    throws PropertyConfigurationException {
        String lbPolicyClassname = lbPolicyClass.getName();

        if(propertyData.containsKey(lbPolicyClassname)) {
            return;
        }

        ConcurrentHashMap<String, LBPropertyData> properties =
            new ConcurrentHashMap<String, LBPropertyData>();

        for(Method method : lbPolicyClass.getDeclaredMethods()) {
            String methodName = method.getName();

            if(methodName.startsWith("get") && !methodName.equals("getLBStrategy")) {
                LBPropertyData lbPropertyData =
                    new LBPropertyData(method, lbPolicyClass);
                properties.putIfAbsent(
                        lbPropertyData.getLbPropertyName(), lbPropertyData);
            }
        }
        propertyData.putIfAbsent(lbPolicyClassname, properties);
    }

    /**
     * Returns a pure policy for a given class.
     * @param lbPolicyClass a class defining a LB policy
     * @return a pure policy to access to the object with the given name
     * @throws LBPolicyFactoryException if the policy cannot be constructed
     */
    @SuppressWarnings("unchecked")
    public ILBPolicy<T> createLBPolicy(final Class<? extends ILBPolicy> lbPolicyClass) throws LBPolicyFactoryException {

        ILBPolicy<T> lbPolicy = null;

        // Lookup if it exists a constructor with a parameter of type ClusterviewManager
        Constructor<?>[] constructors = lbPolicyClass.getDeclaredConstructors();
        for(Constructor<?> constructor : constructors) {
            Class<?>[] paramTypes = constructor.getParameterTypes();
            if(paramTypes.length == 1 && paramTypes[0].equals(ClusterViewManager.class)) {
                try {
                    lbPolicy = (ILBPolicy<T>) constructor.newInstance(clusterViewManager);
                } catch (Exception e) {
                    LOGGER.error("Cannot construct the LB policy with the instance of ClusterViewManager", e);
                    throw new LBPolicyFactoryException(
                            "Cannot construct the LB policy with the instance of ClusterViewManager", e);
                }
                break;
            }
        }
        // Use the default constructor
        if(lbPolicy == null) {
            try {
                lbPolicy = (ILBPolicy<T>) lbPolicyClass.newInstance();
            } catch (Exception e) {
                LOGGER.error("Cannot construct the LB policy", e);
                throw new LBPolicyFactoryException("Cannot construct the LB policy", e);
            }
        }
        // Find a setter for cluster view manager
        try {
            Method setter = lbPolicyClass.getMethod("setClusterViewManager", ClusterViewManager.class);
            setter.invoke(lbPolicy, clusterViewManager);
        } catch (Exception e) {
            LOGGER.debug("Cannot set the manager of cluster view", e);
        }
        return lbPolicy;
    }

    /**
     * Sets properties for a given policy.
     * @param lbPolicy a policy
     * @param properties properties of the policy
     * @throws PropertyConfigurationException if a property cannot be set
     */
    private void setProperties(final ILBPolicy<T> lbPolicy, final Map<String, ?> properties)
    throws PropertyConfigurationException {
        for(String propertyName : properties.keySet()) {
            Object propertyValue = properties.get(propertyName);
            setProperty(lbPolicy, propertyName, propertyValue);
        }
    }

    /**
     * Set a property for a given policy.
     * @param lbPolicy a policy
     * @param propertyName a name of property
     * @param propertyValue a value of property
     * @throws PropertyConfigurationException
     */
    @SuppressWarnings("unchecked")
    private void setProperty(
            final ILBPolicy<T> lbPolicy, final String propertyName, final Object propertyValue)
    throws PropertyConfigurationException {
        Class<? extends ILBPolicy> lbPolicyClass = lbPolicy.getClass();
        LBPropertyData lbPropertyData =
            propertyData.get(lbPolicyClass.getName()).get(propertyName);
        try {
            lbPropertyData.getSetter().invoke(lbPolicy, propertyValue);
        } catch (Exception e) {
            LOGGER.error("Cannot set the property for name {0} with {1}",
                    propertyName, propertyValue, e);
            throw new PropertyConfigurationException(
                    "Cannot set the property for name "
                    + propertyName + " with "
                    + propertyValue, e);
        }
    }

    /**
     * Returns a LB strategy for a given class.
     * @param lbStrategyClass a class defining a LB strategy
     * @return a strategy to access to the object with the given name
     * @throws LBPolicyFactoryException if the strategy cannot be constructed
     * @throws ObjectNotFoundException if none object has the given name
     */
    @SuppressWarnings("unchecked")
    private ILBStrategy<T> createLBStrategy(final Class<? extends ILBStrategy> lbStrategyClass)
    throws LBPolicyFactoryException {

        ILBStrategy<T> lbStrategy = null;

        // Lookup if it exists a constructor with a parameter of type ClusterviewManager
        Constructor<?>[] constructors = lbStrategyClass.getDeclaredConstructors();
        for(Constructor<?> constructor : constructors) {
            Class<?>[] paramTypes = constructor.getParameterTypes();
            if(paramTypes.length == 1 && paramTypes[0].equals(ClusterViewManager.class)) {
                try {
                    lbStrategy = (ILBStrategy<T>) constructor.newInstance(clusterViewManager);
                } catch (Exception e) {
                    LOGGER.error("Cannot construct the LB strategy with the instance of ClusterViewManager", e);
                    throw new LBPolicyFactoryException(
                            "Cannot construct the LB strategy with the instance of ClusterViewManager", e);
                }
                break;
            }
        }

        // Use the default constructor
        if(lbStrategy == null) {
            try {
                lbStrategy = (ILBStrategy<T>) lbStrategyClass.newInstance();
            } catch (Exception e) {
                LOGGER.error("Cannot construct the LB strategy with the default constructor", e);
                throw new LBPolicyFactoryException("Cannot construct the LB strategy with the default constructor", e);
            }
        }
        return lbStrategy;
    }

    //-------------------- Access to lb properties ---------------------------------------------------
    // TODO Should be externalized in a separate class

    @SuppressWarnings("unchecked")
    public static Map<String, Object> getProperties(final ILBPolicy<?> lbPolicy)
    throws PropertyConfigurationException {
        Class<? extends ILBPolicy> lbPolicyClass = lbPolicy.getClass();
        // Check if properties have initialized
        findProperties(lbPolicyClass);
        Map<String, LBPropertyData> lbPropertyData =
            propertyData.get(lbPolicyClass.getName());
        Map<String, Object> properties = new HashMap<String, Object>();
        Object propertyValue;
        for(String propertyName : lbPropertyData.keySet()) {
            propertyValue = getProperty(lbPolicy, propertyName);
            properties.put(propertyName, propertyValue);
        }
        return properties;
    }

    @SuppressWarnings("unchecked")
    public static Object getProperty(final ILBPolicy<?> lbPolicy, final String propertyName) throws PropertyConfigurationException {
        Class<? extends ILBPolicy> lbPolicyClass = lbPolicy.getClass();
        // Check if properties have initialized
        findProperties(lbPolicyClass);
        Map<String, LBPropertyData> lbPropertyData =
            propertyData.get(lbPolicyClass.getName());
        LBPropertyData property = lbPropertyData.get(propertyName);
        try {
            return property.getGetter().invoke(lbPolicy);
        } catch (Exception e) {
            LOGGER.error("Cannot invoke the getter for property {0}", propertyName, e);
            return null;
        }
    }

    @SuppressWarnings("unchecked")
    public static Type getPropertyRawType(
            final Class<? extends ILBPolicy> lbPolicyClass, final String propertyName)
    throws PropertyConfigurationException {
        Type propertyType = getPropertyType(lbPolicyClass, propertyName);
        if(propertyType instanceof ParameterizedType) {
            return ((ParameterizedType) propertyType).getRawType();
        } else {
            return propertyType;
        }
    }

    @SuppressWarnings("unchecked")
    public static Type getPropertyType(
            final Class<? extends ILBPolicy> lbPolicyClass, final String propertyName) throws PropertyConfigurationException {
        // Check if properties have initialized
        findProperties(lbPolicyClass);
        return propertyData.get(lbPolicyClass.getName()).get(propertyName).getLbPropertyType();
    }

    @SuppressWarnings("unchecked")
    public static Map<String, LBPropertyData> getPropertyData(
            final Class<? extends ILBPolicy> lbPolicyClass)
    throws PropertyConfigurationException {
        // Check if properties have initialized
        findProperties(lbPolicyClass);
        return propertyData.get(lbPolicyClass.getName());
    }

    /**
     * Converts the given value from String to the given type.
     * @param lbPropertyName
     * @param svalue a value
     * @return an object that has the same that the given field
     * @throws PropertyConfigurationException if the conversion is not possible
     */
    @SuppressWarnings("unchecked")
    public static Object convertString(
            final Class<? extends ILBPolicy> lbPolicyClass,
            final String lbPropertyName,
            final String svalue) throws PropertyConfigurationException {
        Type propertyType = getPropertyType(lbPolicyClass, lbPropertyName);
        return convertString(propertyType, svalue);
    }

    public static Object convertString(
            final Type propertyType,
            final String svalue) throws PropertyConfigurationException {
        if(propertyType instanceof ParameterizedType) {
            LOGGER.error("Cannot convert a parameterized type.");
            throw new PropertyConfigurationException("Cannot convert a parameterized type");
        }
        Class<?> klass = (Class<?>) propertyType;
        Object propertyValue;
        //  Attempt to cast String forward a primitive type
        if(klass.equals(String.class)) {
            propertyValue = svalue;
        } else if(klass.equals(boolean.class)) {
            propertyValue = Boolean.valueOf(svalue);
        } else if(klass.equals(byte.class)) {
            propertyValue = Byte.valueOf(svalue);
        } else if(klass.equals(int.class)) {
            propertyValue = Integer.valueOf(svalue);
        } else if(klass.equals(long.class)) {
            propertyValue = Long.valueOf(svalue);
        } else if(klass.equals(float.class)) {
            propertyValue = Float.valueOf(svalue);
        } else if(klass.equals(double.class)) {
            propertyValue = Double.valueOf(svalue);
        } else if(klass.equals(URL.class)){ // Try with some complex type
            try {
                propertyValue = new URL(svalue);
            } catch (MalformedURLException e) {
                throw new PropertyConfigurationException("Cannot convert the string " + svalue + " to an URL", e);
            }
        } else if(klass.equals(File.class)) {
            propertyValue = new File(svalue);
        } else if(klass.equals(Class.class)) {
            try {
                propertyValue = Class.forName(svalue);
            } catch (ClassNotFoundException e) {
                throw new PropertyConfigurationException("Cannot convert the string " + svalue + " to a class", e);
            }
        } else {
            LOGGER.error("Type of field not supported: {0}", klass.getName());
            throw new PropertyConfigurationException(
                    "Type of field not supported: " + klass.getName());
        }
        return propertyValue;
    }

    @SuppressWarnings("unchecked")
    public static List<?> convertStrings(
            final Class<? extends ILBPolicy> lbPolicyClass,
            final String lbPropertyName,
            final List<String> slist)throws PropertyConfigurationException {
        Type parameterizedType = getPropertyType(lbPolicyClass, lbPropertyName);
        if(!(parameterizedType instanceof ParameterizedType)) {
            LOGGER.error("{0} is not a parameterized type: cannot convert the elements of this collection", parameterizedType);
            throw new PropertyConfigurationException(
                    parameterizedType + "is not a parameterized type: cannot convert the elements of this collection");
        }
        Type propertyType = ((ParameterizedType) parameterizedType).getActualTypeArguments()[0];
        List<Object> values = new ArrayList<Object>();
        for(String svalue : slist) {
            values.add(convertString(propertyType, svalue));
        }
        return values;
    }

}
