/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: LoadFactorComparator.java 1604 2008-01-28 22:14:00Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.strategy;

import java.util.Comparator;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A comparator of load-factors.
 * @param <T> The type of objects that are load-balanced
 * @author The new CMI team
 */
public final class LoadFactorComparator<T extends LoadBalanceable> implements Comparator<T> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(LoadFactorComparator.class);

    /**
     * The manager of the cluster view.
     */
    private final ClusterViewManager clusterViewManager;

    /**
     * Construct a comparator of load-factor.
     * The load-factors are provided by the {@link ClusterViewManager}.
     * @param clusterViewManager the manager of the cluster view
     */
    public LoadFactorComparator(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
    }

    /**
     * Compares the load-factors of two {@link LoadBalanceable} with the load factor of their {@link ServerRef}.
     * @param loadBalanceable1 the first {@link LoadBalanceable}
     * @param loadBalanceable2 the second {@link LoadBalanceable}
     * @return a negative integer, zero, or a positive integer as the first server has a smaller,
     * equal to, or greater load-factor than than the second
     */
    public int compare(final T loadBalanceable1, final T loadBalanceable2) {
        ServerRef serverRef1 = loadBalanceable1.getServerRef();
        ServerRef serverRef2 = loadBalanceable2.getServerRef();
        try {
            int factor1 = clusterViewManager.getLoadFactor(serverRef1);
            int factor2 = clusterViewManager.getLoadFactor(serverRef2);
            return new Integer(factor1).compareTo(factor2);
        } catch (ServerNotFoundException e) {
            LOGGER.error("Cannot compare the load factors of servers {0} and {1}.", serverRef1, serverRef2, e);
            return 0;
        }
    }

}
