/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id:LoadFactor.java 1124 2007-07-27 16:38:35Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.strategy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import net.jcip.annotations.Immutable;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Defines a strategy that orders the list of servers with the load-factor.
 * @param <T> The type of objects that are load-balanced
 * @author The new CMI team
 */
@Immutable
public class LoadFactorSort<T extends LoadBalanceable> implements ILBStrategy<T> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(LoadFactorSort.class);

    /**
     * The comparator of load-factors to sort the list.
     */
    private final LoadFactorComparator<T> comparator;

    /**
     * Constructs a strategy for load-factor.
     * @param clusterViewManager the manager of the cluster view
     */
    public LoadFactorSort(final ClusterViewManager clusterViewManager) {
        comparator = new LoadFactorComparator<T>(clusterViewManager);
    }

    /**
     * Returns a list of {@link LoadBalanceable} ordered by their load factor (to be on the decrease).
     * @param loadBalanceables a list of CMIReference
     * @return a list of CMIReference ordered by load-factor
     */
    public List<T> choose(final List<T> loadBalanceables) {
        List<T> orderedLoadBalanceables = new ArrayList<T>(loadBalanceables);
        Collections.sort(orderedLoadBalanceables, comparator);
        return orderedLoadBalanceables;
    }

    @Override
    public String toString() {
        return "LoadFactor";
    }

}
