/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: LoadFactorWeight.java 1605 2008-01-29 09:35:08Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.lb.strategy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.lb.LoadBalanceable;
import org.ow2.carol.cmi.reference.ServerNotFoundException;
import org.ow2.carol.cmi.reference.ServerRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Defines a strategy that attributes a weight for each load-balanceable.
 * @param <T> The type of objects that are load-balanced
 * @author Loris Bouzonnet
 */
public class LoadFactorWeight<T extends LoadBalanceable> implements ILBStrategy<T> {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(LoadFactorWeight.class);

    /**
     * The comparator of load-factors to sort the list.
     */
    private final LoadFactorComparator<T> comparator;

    private final ClusterViewManager clusterViewManager;

    public LoadFactorWeight(final ClusterViewManager clusterViewManager) {
        this.clusterViewManager = clusterViewManager;
        comparator = new LoadFactorComparator<T>(clusterViewManager);
    }

    public List<T> choose(final List<T> loadBalanceables) {

        List<T> weightedList = new ArrayList<T>();

        List<T> orderedLoadBalanceables = new ArrayList<T>(loadBalanceables);
        Collections.sort(orderedLoadBalanceables, comparator);

        ServerRef maxLoaded =
            orderedLoadBalanceables.get(orderedLoadBalanceables.size() - 1).getServerRef();

        int maxLoad;
        try {
            maxLoad = clusterViewManager.getLoadFactor(maxLoaded);
        } catch (ServerNotFoundException e) {
            LOGGER.error("Cannot get the load factor for {0}. The strategy cannot be applied.", maxLoaded, e);
            return loadBalanceables;
        }

        LOGGER.debug("The max load factor {0} is for {1}", maxLoad, maxLoaded);

        // Normalize (percentage) and attribute the corresponding weights
        for(T loadBalanceable : orderedLoadBalanceables) {
            int weight;
            try {
                weight = 100 - Math.round(
                        clusterViewManager.getLoadFactor(loadBalanceable.getServerRef()) / maxLoad * 100);
            } catch (ServerNotFoundException e) {
                LOGGER.error("Cannot get the load factor for {0}. The strategy cannot be applied.",
                        loadBalanceable, e);
                return loadBalanceables;
            }
            for(int i = 0 ; i < weight ; i++) {
                weightedList.add(loadBalanceable);
            }

        }

        return weightedList;
    }

}
