/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:CmiInvocationHandler.java 949 2007-06-02 17:24:33Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.rpc;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * This is the standard {@link InvocationHandler} of a {@link CMIProxy}.
 * It is created by {@link CMIProxyFactory#newCMIProxy(ClusterViewManager, String, String)}.
 * The {@link CMIProxy} intercepts the invocation of method of the clustered stateless POJOs.
 * @author The new CMI team
 * @see CMIProxyFactory
 */
@ThreadSafe
public class POJOInvocationHandler extends CMIInvocationHandler {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(POJOInvocationHandler.class);

    /**
     * The classloader to use.
     * When the smart factory is used, it is the smart classloader.
     */
    private final ClassLoader classLoader;

    /**
     * Interface of the object.
     */
    private final Class<?> itf;

    /**
     * Build a new CMI Invocation Handler.
     * @param classLoader the classloader to use (when the smart factory is used, it is the smart classloader)
     * @param clusterViewManager A manager of the cluster view
     * @param objectName a name of the object
     * @param protocolName a protocol to perform the invocation
     * @param itf interface of the object
     */
    public POJOInvocationHandler(
            final ClassLoader classLoader,
            final ClusterViewManager clusterViewManager,
            final String objectName,
            final String protocolName,
            final Class<?> itf) {
        super(clusterViewManager, objectName, protocolName, false);
        this.classLoader = classLoader;
        this.itf = itf;
    }

    /**
     * Handles remote methods.
     * @throws CMIInvocationHandlerException if a server cannot be chosen
     * @throws Throwable if the invocated method throws an exception
     **/
    @Override
    protected Object invokeRemoteMethod(final Object proxy, final Method method, final Object[] args)
    throws POJOInvocationHandlerException, Throwable {

        // Use the same classloader that during the creation of this object
        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        setClassLoader(oldClassLoader);
        try {
            return super.invokeRemoteMethod(proxy, method, args);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    }

    /**
     * Returns a string representation for a proxy that uses this invocation handler.
     **/
    @Override
    protected String proxyToString(final Object proxy) {
        return "POJOProxy["+super.proxyToString(proxy)+", itf:"+itf.getName()+"]";
    }

}
