/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:CmiInvocationHandler.java 949 2007-06-02 17:24:33Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.rpc;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import net.jcip.annotations.ThreadSafe;

import org.ow2.carol.cmi.controller.common.ClusterViewManager;
import org.ow2.carol.cmi.pool.StubOrProxyFactory;
import org.ow2.carol.cmi.reference.CMIProxyHandle;
import org.ow2.carol.cmi.reference.CMIProxyHandleImpl;
import org.ow2.carol.cmi.reference.CMIReference;
import org.ow2.carol.cmi.reference.CMIReferenceable;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.pool.impl.JPool;

/**
 * This is the standard {@link InvocationHandler} of a {@link CMIProxy}.
 * It is created by {@link CMIProxyFactory#newCMIProxy(ClusterViewManager, String, String)}.
 * The {@link CMIProxy} intercepts the invocation of method of the clustered stateless POJOs.
 * @author The new CMI team
 * @see CMIProxyFactory
 */
@ThreadSafe
public class POJOInvocationHandler extends CMIInvocationHandler<Object> {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = 4461812198372895346L;

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(POJOInvocationHandler.class);

    /**
     * The classloader to use.
     * When the smart factory is used, it is the smart classloader.
     */
    private transient ClassLoader classLoader;

    /**
     * Build a new CMI Invocation Handler.
     * @param classLoader the classloader to use (when the smart factory is used, it is the smart classloader)
     * @param clusterViewManager A manager of the cluster view
     * @param objectName a name of the object
     * @param protocolName a protocol to perform the invocation
     * @param itf interface of the object
     */
    public POJOInvocationHandler(
            final ClassLoader classLoader,
            final ClusterViewManager clusterViewManager,
            final String objectName,
            final String protocolName,
            final Class<?> itf) {
        super(clusterViewManager, objectName, protocolName, false, itf);
        this.classLoader = classLoader;
    }

    /**
     * Handles remote methods.
     * @throws CMIInvocationHandlerException if a server cannot be chosen
     * @throws Throwable if the invoked method throws an exception
     **/
    @Override
    protected Object invokeRemoteMethod(final Object proxy, final Method method, final Object... args)
    throws POJOInvocationHandlerException, Throwable {

        // Use the same classloader that during the creation of this object
        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        setClassLoader(oldClassLoader);
        try {
            return super.invokeRemoteMethod(proxy, method, args);
        } finally {
            Thread.currentThread().setContextClassLoader(oldClassLoader);
        }
    }

    /**
     * Returns a string representation for a proxy that uses this invocation handler.
     **/
    @Override
    protected String proxyToString(final Object proxy) {
        return "POJOProxy[" + super.proxyToString(proxy) + "]";
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void checkInitialized() throws POJOInvocationHandlerException {
        try {
            synchronized(clusterViewManager){
                // The first time, create a pool to share stubs for EJBHome
                if(clusterViewManager.getPool(objectName) == null) {
                    LOGGER.debug("First lookup on {0}: creation of the pool for its stubs for EJBHome", objectName);
                    clusterViewManager.setPool(
                            objectName,
                            new JPool<CMIReferenceable, CMIReference>(
                                    new StubOrProxyFactory(clusterViewManager)));
                }
            }
        } catch (Exception e) {
            LOGGER.error("Cannot init the pool", e);
            throw new POJOInvocationHandlerException("Cannot init the pool", e);
        }
        if(classLoader == null) {
            try {
                classLoader = Thread.currentThread().getContextClassLoader();
            } catch(Exception e) {
                throw new POJOInvocationHandlerException(
                        "Cannot retrieve the classloader for object with name " + objectName, e);
            }
        }
    }

    @Override
    protected CMIProxyHandle getHandle(final CMIProxy cmiProxy) {
        if(cmiProxyHandle == null) {
            cmiProxyHandle = new CMIProxyHandleImpl(objectName, itf.getName(), cmiProxy);
        }
        return cmiProxyHandle;
    }

}
