
/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: ProviderURLsAnswer.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.smart.message;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.ow2.carol.cmi.smart.api.Message;
import org.ow2.carol.cmi.smart.api.Operations;
import org.ow2.carol.cmi.smart.server.SmartEndPoint;


/**
 * Server sends the list of provider urls to client. The new CMI team
 */
public class ProviderURLsAnswer implements Message {

    /**
     * Logger.
     */
    private static Logger logger = Logger.getLogger(SmartEndPoint.class
            .getName());

    /**
     * List of provider url.
     */
    private List<String> providerURLs = null;

    /**
     * The length of an integer.
     */
    private static int intlength = 4;

    /**
     * Create a new answer with the given buffer for the list of provider.
     * url.
     *
     * @param buffer the given buffer.
     */
    @SuppressWarnings("unchecked")
    public ProviderURLsAnswer(final ByteBuffer buffer) {

        // Get the total number of providers
        ArrayList<String> urls = new ArrayList<String>();
        int size = buffer.getInt(buffer.position());
        int offset = 0;

        for (int i = 0; i < size; i++) {

            // Get the length of the current provider url.
            ByteBuffer newbuf = buffer.duplicate();
            int urlsize = newbuf.getInt(buffer.position()+intlength+offset);

            // Get the ByteBuffer of the current provider url.//
            newbuf.limit(buffer.position()+intlength+offset+intlength+urlsize);
            newbuf.position(buffer.position()+intlength + offset + intlength);

            // Decode the current provider url.
            ByteBuffer urlbuf = newbuf.duplicate();
            String url = decode(urlbuf);

            // Put the current provider url into the url list.
            urls.add(url);

            offset = urlsize + intlength;
        }
        this.providerURLs = urls;
    }

    /**
     * Create a new answer with the given list of provider url.
     * @param providerURLs the given list of provider url
     */
    public ProviderURLsAnswer(final List<String> providerURLs) {
        this.providerURLs = providerURLs;
    }

    /**
     * Get the message to send.
     * @return the bytebuffer to send.
     */
    public ByteBuffer getMessage() {
        ByteBuffer message = getMessageContent();

        // compute length
        int length = HEADER_SIZE;
        if (message != null) {
            length += message.capacity();
        }

        // Create ByteBuffer
        ByteBuffer byteBuffer = ByteBuffer.allocate(length);

        // Append header
        byteBuffer.put(getOperationCode());

        if (message != null) {
            byteBuffer.putInt(message.capacity());
        }

        // append inner message (go to position 0 first)
        if (message != null) {
            message.position(0);
            byteBuffer.put(message);
        }

        // reset our position
        byteBuffer.position(0);

        // return buffer
        return byteBuffer;
    }

    /**
     * Gets the Operation Code of this answer.
     * @return the operation code.
     */
    public byte getOperationCode() {
        return Operations.PROVIDER_URLS_ANSWER;
    }

    /**
     * Gets the content of this message (only this part, not the header).
     *
     * @return the content of this message.
     */
    public ByteBuffer getMessageContent() {
        if (providerURLs != null) {
            // Transform the list de provider url to string array
            String[] urlsArray = new String[providerURLs.size()];
            for (int n = 0; n < providerURLs.size(); n++) {
                urlsArray[n] = providerURLs.toArray()[n].toString();
            }

            // Create ByteBuffer
            int length = getMessageLength(providerURLs);
            ByteBuffer buffer = ByteBuffer.allocate(length);

            // Put the total number of providers in the head of ByteBuffer
            buffer.putInt(providerURLs.size());

            // Put every provider url with its length in the buffer
            for (int i = 0; i < providerURLs.size(); i++) {

                // Put the length and the content of each url into the buffer
                buffer.putInt(urlsArray[i].length());
                ByteBuffer urlbuffer = ByteBuffer.allocate(urlsArray[i].getBytes().length);
                urlbuffer = encode(urlsArray[i]);
                urlbuffer.position(0);
                buffer.put(urlbuffer);
            }

            buffer.position(0);

            return buffer;
        } else {
            logger
                    .log(Level.INFO,
                            "The providers url List should not be null.");
            return null;
        }
    }

    /**
     * Get the length of message content.
     *
     * @param providerURLs
     *            the message content.
     * @return the content length.
     */
    private int getMessageLength(final List<String> providerURLs) {
        int length = 0;

        if (providerURLs != null) {
            int size = providerURLs.size();
            for (int i = 0; i < size; i++) {
                length += providerURLs.toArray()[i].toString().getBytes().length
                        + intlength;
            }
            return (length + intlength);
        } else {
            logger.log(Level.INFO, "The given List should not be null.");
            return 0;
        }
    }

    /**
     * Get the list of provider url.
     * @return the list of provider url.
     */
    public List<String> getProviderURLs() {
        return this.providerURLs;
    }

    /**
     * Encode the given string into a bytebuffer.
     *
     * @param str
     *            the given string
     * @return a bytebuffer with UTF-8 encoded string
     */
    protected ByteBuffer encode(final String str) {
        byte[] bytes = null;
        try {
            bytes = str.getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        ByteBuffer buffer = ByteBuffer.allocate(bytes.length);

        buffer.put(bytes);

        return buffer;

    }

    /**
     * Decode the string encoded in the bytebuffer in UTF-8 format.
     *
     * @param buffer
     *            the given buffer to analyze.
     * @return the decoded string
     */
    protected String decode(final ByteBuffer buffer) {
        Charset charset = Charset.forName("UTF-8");
        CharsetDecoder charsetDecoder = charset.newDecoder();

        CharBuffer charBuffer = null;
        try {
            charBuffer = charsetDecoder.decode(buffer);
        } catch (CharacterCodingException e) {
            throw new IllegalStateException("Invalid characted encoding", e);
        }
        return charBuffer.toString();
    }

}
