/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: SmartEndPoint.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.smart.server;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.ow2.carol.cmi.smart.api.Message;
import org.ow2.carol.cmi.smart.api.Operations;
import org.ow2.carol.cmi.smart.message.Attachment;
import org.ow2.carol.cmi.smart.message.ClassAnswer;
import org.ow2.carol.cmi.smart.message.ClassRequest;
import org.ow2.carol.cmi.smart.message.ClassNotFound;
import org.ow2.carol.cmi.smart.message.ResourceNotFound;
import org.ow2.carol.cmi.smart.message.FactoryNameAnswer;
import org.ow2.carol.cmi.smart.message.ProviderURLsAnswer;
import org.ow2.carol.cmi.smart.message.ResourceAnswer;
import org.ow2.carol.cmi.smart.message.ResourceRequest;
import org.ow2.carol.cmi.smart.api.SmartConnector;

/**
 * This class receives the request from clients, handle them and send an answer.
 *
 * @author The new CMI team
 * @author Florent Benoit
 */

public class SmartEndPoint implements Runnable {

    /**
     * Logger.
     */
    private static Logger logger = Logger.getLogger(SmartEndPoint.class
            .getName());

    /**
     * Maximum length of messages that server accepts.
     */
    private static final int MAX_LENGTH_INCOMING_MSG = 2048;

    /**
     * Buffer length.
     */
    private static final int BUFFER_LENGTH = 3000;

    /**
     * Default port number.
     */
    private static final int DEFAULT_PORT_NUMBER = 2505;

    /**
     * Listening port number.
     */
    private int portNumber = DEFAULT_PORT_NUMBER;

    /**
     * Nio Selector.
     */
    private Selector selector = null;

    /**
     * Server socket channel (listening).
     */
    private ServerSocketChannel serverSocketChannel = null;

    /**
     * server selection key(accepting clients).
     */
    private SelectionKey selectionKey = null;

    /**
     * started for listening.
     */
    private boolean started = false;

    /**
     * Used to get the list of provider urls and the factory name.
     */
    private SmartConnector connector = null;

    /**
     * Start method.
     *
     * @throws SmartServerException
     *             if the start has failed.
     */
    public void start() throws SmartServerException {

        // open a new selector
        try {
            selector = Selector.open();
        } catch (IOException e) {
            logger.log(Level.INFO, "Cannot open a new selector.");
            throw new SmartServerException("Cannot open a new selector.", e);
        }

        // open a new server socket channel
        try {
            serverSocketChannel = ServerSocketChannel.open();
            serverSocketChannel.configureBlocking(false);
        } catch (IOException e) {
            logger.log(Level.INFO, "Cannot open a new server socket channel.");
            throw new SmartServerException(
                    "Cannot open a new server socket channel.", e);
        }

        // define port number listener
        try {
            // create a socket address
            InetSocketAddress endpoint = new InetSocketAddress(portNumber);
            // Retrieves a server socket associated with this channel
            // Binds the ServerSocket to a specific address
            serverSocketChannel.socket().bind(endpoint);
        } catch (IOException e) {
            logger
                    .log(Level.INFO, "Port Number" + portNumber
                            + "can't be used");
            throw new SmartServerException("Port Number" + portNumber
                    + "can't be used", e);
        }

        // Registers this channel with the given selector, returning a selection
        // key.
        try {
            selectionKey = serverSocketChannel.register(selector,
                    SelectionKey.OP_ACCEPT);
        } catch (ClosedChannelException e) {
            logger.log(Level.INFO, "Cannot select this server as an listener");
            throw new SmartServerException(
                    "Cannot select this server as an listener", e);
        }

        // start server for listening
        started = true;

        new Thread(this).start();

        logger.log(Level.INFO, "SmartClient Endpoint listening on port '"
                + portNumber + "'.");
    }

    /**
     * Get une liste de providerURL.
     *
     * @param protocol
     *            The protocol used in the client side.
     * @return une liste de providerURL
     */
    private List<String> getproviderURLs(final String protocol) {
        List<String> providerURLs = null;
        try {
            providerURLs = connector.getProviderURLs(protocol);
            return providerURLs;
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return providerURLs;
        }
    }

    /**
     * Get the factory name with the given protocol.
     *
     * @param protocol
     *            The protocol used in the client side.
     * @return The factory name.
     */
    private String getFactoryName(final String protocol) {
        String factoryName = connector.getInitialContextFactoryName(protocol);
        return factoryName;
    }

    /**
     * Start the thread for looking at the selectors.
     */
    public void run() {
        selectorHandle();
    }

    /**
     * Infinite loop (until the end of the component) that handle the selectors.
     */
    private void selectorHandle() {
        while (started == true) {
            // Selects a set of keys whose corresponding channels are ready for
            // I/O operations.
            int keys = 0;
            try {
                keys = selector.select();
            } catch (IOException e) {
                logger.log(Level.INFO, "this selector is closed", e);
                started = false;
            }

            if (keys == 0) {
                continue;
            }

            // gets this selector's selected-key set.
            Set<SelectionKey> selectedKeys = selector.selectedKeys();
            for (Iterator<SelectionKey> isSelectedKeys = selectedKeys
                    .iterator(); isSelectedKeys.hasNext();) {
                SelectionKey selectKey = isSelectedKeys.next();
                isSelectedKeys.remove(); // remove it

                // verification if it is the server key ?
                if (selectKey == selectionKey) {
                    // Tests whether this key's channel is ready to accept a new
                    // socket connection.
                    if (selectKey.isAcceptable()) {
                        readyHandle();
                    }
                }
                // Tests whether this key's channel is ready for reading.
                else if (selectKey.isReadable()) {
                    // read request from the client

                    try {
                        readHandle(selectKey);
                    } catch (IOException e) {
                        logger.log(Level.INFO,
                                "Unable to read request data from the client.",
                                e);
                    }
                }
                // Tests whether this key's channel is ready for writing.
                else if (selectKey.isWritable()) {
                    // write answer to the client
                    try {
                        writeHandle(selectKey);
                    } catch (IOException e) {
                        logger
                                .log(
                                        Level.FINE,
                                        "Unable to write answer data to the client.",
                                        e);
                    }
                }

            }
        }
    }

    /**
     * Handle a new connected client.
     */
    private void readyHandle() {
        // Accepts a connection made to this channel's socket.
        try {
            SocketChannel clientConnection = serverSocketChannel.accept();

            // no blocking client
            clientConnection.configureBlocking(false);

            // Register client (with an empty channel attachment)
            clientConnection.register(selector, SelectionKey.OP_READ,
                    new Attachment());

        } catch (IOException e) {
            logger.log(Level.FINE, "this channel is closed.", e);
        }
    }

    /**
     * Handle all write operations on channels.
     *
     * @param selectKey
     *            The selected key.
     * @throws IOException
     *             if cannot write to the channel.
     */
    private void writeHandle(final SelectionKey selectKey) throws IOException {
        SocketChannel serverConnection = (SocketChannel) selectKey.channel();

        // Write the data that was attached on the selection key
        ByteBuffer byteBuffer = (ByteBuffer) selectKey.attachment();

        if (byteBuffer.hasRemaining()) {
            serverConnection.write(byteBuffer);
        } else {
            // finished to write, close
            serverConnection.close();
        }
    }

    /**
     * Handle all read operations on channels.
     *
     * @param selectKey
     *            the selected key.
     * @throws IOException
     *             if cannot read from the channel.
     */
    private void readHandle(final SelectionKey selectKey) throws IOException {
        // Returns the channel for which this key was created.
        SocketChannel clientConnection = (SocketChannel) selectKey.channel();
        // Retrieves the current attachment
        Attachment attachment = (Attachment) selectKey.attachment();
        // Current bytecode read
        ByteBuffer byteBuffer = attachment.getByteBuffer();
        // Read again
        int bytes = clientConnection.read(byteBuffer);
        if (bytes == -1) {
            // close (as the client has been disconnected)
            // Requests that the registration of this key's
            // channel with its selector be cancelled.
            selectKey.cancel();
            clientConnection.close();
        }

        // analyse client data
        if (byteBuffer.position() >= Message.HEADER_SIZE) {

            // Get operation asked by client
            byte operationCode = byteBuffer.get(0);

            // Length
            int length = byteBuffer.getInt(1);
            if (length < 0) {
                selectionKey.cancel();
                clientConnection.close();
                throw new IllegalStateException("Invalid length for client '"
                        + length + "'.");
            }

            if (length > MAX_LENGTH_INCOMING_MSG) {
                selectionKey.cancel();
                clientConnection.close();
                throw new IllegalStateException(
                        "Length too big, max length = '"
                                + MAX_LENGTH_INCOMING_MSG + "', current = '"
                                + length + "'.");
            }

            // Correct header and correct length ?
            if (byteBuffer.position() >= Message.HEADER_SIZE + length) {
                // set the limit (specified in the length), else we have a
                // default buffer limit
                byteBuffer.limit(Message.HEADER_SIZE + length);

                // duplicate this buffer
                ByteBuffer dataBuffer = byteBuffer.duplicate();

                // skip header (already analyzed)
                dataBuffer.position(Message.HEADER_SIZE);

                // Switch on operations :
                try {
                    switch (operationCode) {
                    case Operations.CLASS_REQUEST:
                        classRequest(selectKey, dataBuffer);
                        break;
                    case Operations.RESOURCE_REQUEST:
                        resourceRequest(selectKey, dataBuffer);
                        break;
                    case Operations.PROVIDER_URLS_REQUEST:
                        providerURLsRequest(selectKey, dataBuffer);
                        break;
                    case Operations.FACTORY_NAME_REQUEST:
                        factoryNameRequest(selectKey, dataBuffer);
                        break;
                    default:
                        // nothing to do
                    }
                } catch (Exception e) {
                    // clean
                    selectionKey.cancel();
                    clientConnection.close();
                    throw new IllegalStateException(
                            "Cannot handle request with opCode '"
                                    + operationCode + "'.", e);
                }
            }
        }
    }

    /**
     * Handle the client's provider urls asking.
     *
     * @param selectKey
     *            the selected key.
     * @param dataBuffer
     *            the buffer that contains request.
     */
    private void providerURLsRequest(final SelectionKey selectKey,
            final ByteBuffer dataBuffer) {
        String protocol = null;
        // Decode the protocol used in the client side
        Charset charset = Charset.forName("UTF-8");
        CharsetDecoder decoder = charset.newDecoder();
        CharBuffer charBuffer;
        try {
            charBuffer = decoder.decode(dataBuffer);
            protocol = charBuffer.toString();
        } catch (CharacterCodingException e) {
            throw new IllegalStateException("Invalid characted encoding", e);
        }

        // Answer to the client (go in write mode)
        selectKey.interestOps(SelectionKey.OP_WRITE);

        // Get the list of provider url avec the given protocole
        List<String> providerURLs = getproviderURLs(protocol);

        // Create answer object
        ProviderURLsAnswer providerURLAnswer = new ProviderURLsAnswer(
                providerURLs);

        // Attach the answer on the key
        selectKey.attach(providerURLAnswer.getMessage());

    }

    /**
     * Handle the client's factory name asking.
     *
     * @param selectKey
     *            the selected key.
     * @param dataBuffer
     *            the buffer that contains request.
     */
    private void factoryNameRequest(final SelectionKey selectKey,
            final ByteBuffer dataBuffer) {
        String protocol = null;
        // Decode the protocol used in the client side
        Charset charset = Charset.forName("UTF-8");
        CharsetDecoder decoder = charset.newDecoder();
        try {
            CharBuffer charBuffer = decoder.decode(dataBuffer);
            protocol = charBuffer.toString();
        } catch (CharacterCodingException e) {
            throw new IllegalStateException("Invalid characted encoding", e);
        }

        // Answer to the client (go in write mode)
        selectKey.interestOps(SelectionKey.OP_WRITE);
        // Get the factory name
        String factoryName = getFactoryName(protocol);
        // Create answer object
        FactoryNameAnswer factoryNameAnswer = new FactoryNameAnswer(factoryName);
        // Attach the answer on the key
        selectKey.attach(factoryNameAnswer.getMessage());
    }

    /**
     * Handle the client's request asking for a resource.
     *
     * @param selectKey
     *            key for exchanging with the client.
     * @param dataBuffer
     *            the buffer that contains request.
     * @throws IOException
     *             if operation fails.
     */
    private void resourceRequest(final SelectionKey selectKey,
            final ByteBuffer dataBuffer) throws IOException {
        // Build object (from input)
        ResourceRequest resourceRequest = new ResourceRequest(dataBuffer);
        String resourceName = resourceRequest.getResourceName();

        // Answer to the client
        selectKey.interestOps(SelectionKey.OP_WRITE);

        // Find the resource from the classloader
        URL url = Thread.currentThread().getContextClassLoader().getResource(
                resourceName);
        if (url == null) {
            ResourceNotFound resourceNotFound = new ResourceNotFound(
                    resourceName);
            selectKey.attach(resourceNotFound.getMessage());
            logger.log(Level.FINE, "Resource '" + resourceName + "' not found");
            return;
        }

        InputStream inputStream = url.openStream();

        byte[] bytes = null;
        try {
            // Get bytecode of the class
            bytes = readClass(inputStream);

        } finally {
            inputStream.close();
        }

        // Create answer object
        ResourceAnswer resourceAnswer = new ResourceAnswer(resourceName, bytes);

        // Attach the answer on the key
        selectKey.attach(resourceAnswer.getMessage());

    }

    /**
     * Handle the client's request asking for a class.
     *
     * @param selectKey
     *            key for exchanging with the client.
     * @param dataBuffer
     *            the buffer that contains request.
     */
    private void classRequest(final SelectionKey selectKey,
            final ByteBuffer dataBuffer) {

        ClassRequest classRequest = new ClassRequest(dataBuffer);
        String className = classRequest.getClassName();

        // Answer to the client (go in write mode)
        selectKey.interestOps(SelectionKey.OP_WRITE);
        String encodedClassName = className.replaceAll("\\.", "/").concat(
                ".class");

        // Find the resource from the classloader
        InputStream inputStream = Thread.currentThread()
                .getContextClassLoader().getResourceAsStream(encodedClassName);

        if (inputStream == null) {
            ClassNotFound classNotFound = new ClassNotFound(className);
            selectKey.attach(classNotFound.getMessage());
            logger.log(Level.FINE, "Class '" + className + "' not found");
            return;
        }
        byte[] bytes = null;
        try {
            // Get bytecode of the class
            try {
                bytes = readClass(inputStream);
            } catch (IOException e) {
                e.printStackTrace();
            }
        } finally {
            try {
                inputStream.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        // Create answer object
        ClassAnswer classAnswer = new ClassAnswer(className, bytes);

        // Attach the answer on the key
        selectKey.attach(classAnswer.getMessage());

    }

    /**
     * Gets the bytes from the given input stream.
     *
     * @param is
     *            given input stream.
     * @return The array of bytes for the given input stream.
     * @throws IOException
     *             if class cannot be read.
     */
    private static byte[] readClass(final InputStream is) throws IOException {
        if (is == null) {
            throw new IOException("Given input stream is null");
        }
        byte[] b = new byte[is.available()];
        int len = 0;
        while (true) {
            int n = is.read(b, len, b.length - len);
            if (n == -1) {
                if (len < b.length) {
                    byte[] c = new byte[len];
                    System.arraycopy(b, 0, c, 0, len);
                    b = c;
                }
                return b;
            }
            len += n;
            if (len == b.length) {
                byte[] c = new byte[b.length + BUFFER_LENGTH];
                System.arraycopy(b, 0, c, 0, len);
                b = c;
            }
        }
    }

    /**
     * Set the smart connector.
     * @param connector the given smart connector.
     */
    public void setSmartConnector(final SmartConnector connector) {
        this.connector = connector;
    }

    /**
     * Set the Port number.
     * @param number the given port number.
     */
    public void setPortNumber(final int number){
        this.portNumber= number;
    }

    /**
     * Stop method.
     *
     * @throws SmartServerException
     *             if the stop is failing.
     */
    public void stop() throws SmartServerException {
        started = false;
        selector.wakeup();
    }


}
