/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: SmartContext.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.smart.spi;

import java.util.Hashtable;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.Name;
import javax.naming.NameClassPair;
import javax.naming.NameParser;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;

/**
 * Context that is used in the SmartFactory.
 */
public class SmartContext implements Context {

    /**
     * Context that is wrapped.
     */
    private Context wrapped;

    /**
     * Classloader to use for all methods.
     */
    private ClassLoader classLoader;

    /**
     * Create a new classloader with the given classloader and the wrapped
     * context for delegation.
     *
     * @param wrapped
     *            The context to wrap.
     * @param classLoader
     *            classloader to use.
     */
    public SmartContext(final Context wrapped, final ClassLoader classLoader) {
        this.wrapped = wrapped;
        this.classLoader = classLoader;
    }

    /**
     * Adds a new environment property to the environment of this context. If
     * the property already exists, its value is overwritten.
     *
     * @param propName
     *            the name of the environment property to add; may not be null
     * @param propVal
     *            the value of the property to add; may not be null
     * @return the previous value of the property, or null if the property was
     *         not in the environment before
     * @throws NamingException
     *             if a naming exception is encountered
     */
    public Object addToEnvironment(final String propName, final Object propVal)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.addToEnvironment(propName, propVal);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Retrieves the named object.
     *
     * @param name
     *            the name of the object to look up
     * @return the object bound to <tt>name</tt>
     * @throws NamingException
     *             if a naming exception is encountered
     */
    public Object lookup(final Name name) throws NamingException {
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.lookup(name);
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }

    /**
     * Retrieves the named object.
     *
     * @param name
     *            the name of the object to look up
     * @return the object bound to name
     * @throws NamingException
     *             if a naming exception is encountered
     */
    public Object lookup(final String name) throws NamingException {
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.lookup(name);
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
    }

    /**
     * Bind a new clustered object.
     *
     * @param name
     *            The name to bind; may not be empty.
     * @param obj
     *            The object to bind; possibly null.
     * @throws NamingException
     *             if a naming exception is encountered.
     */
    public void bind(final Name name, final Object obj) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.bind(name, obj);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Bind a new clustered object.
     *
     * @param name
     *            The name to bind; may not be empty.
     * @param obj
     *            The object to bind; possibly null.
     * @throws NamingException
     *             if a naming exception is encountered.
     */
    public void bind(final String name, final Object obj)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.bind(name, obj);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Retrieves the named object, following links except for the terminal
     * atomic component of the name. If the object bound to name is not a link,
     * returns the object itself.
     *
     * @param name
     *            the name of the object to look up
     * @return the object bound to name, not following the terminal link (if
     *         any).
     * @throws NamingException
     *             if a naming exception is encountered
     */
    public Object lookupLink(final Name name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.lookupLink(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Retrieves the named object, following links except for the terminal
     * atomic component of the name. If the object bound to name is not a link,
     * returns the object itself.
     *
     * @param name
     *            the name of the object to look up
     * @return the object bound to name, not following the terminal link (if
     *         any).
     * @throws NamingException
     *             if a naming exception is encountered
     */
    public Object lookupLink(final String name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.lookupLink(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Binds a name to an object, overwriting any existing binding.
     *
     * @param name
     *            the name to bind; may not be empty
     * @param obj
     *            the object to bind; possibly null
     * @throws NamingException
     *             if a naming exception is encountered
     * @see javax.naming.InvalidNameException
     */
    public void rebind(final Name name, final Object obj)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.rebind(name, obj);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }

    }

    /**
     * Binds a name to an object, overwriting any existing binding.
     *
     * @param name
     *            the name to bind; may not be empty
     * @param obj
     *            the object to bind; possibly null
     * @throws NamingException
     *             if a naming exception is encountered
     * @see javax.naming.InvalidNameException
     */
    public void rebind(final String name, final Object obj)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.rebind(name, obj);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }

    }

    /**
     * Rename an object.
     *
     * @param oldName
     *            the old name; may not be empty
     * @param newName
     *            the new name; may not be empty
     * @throws NamingException
     *             if a naming exception is encountered
     * @see javax.naming.NameException
     */
    public void rename(final Name oldName, final Name newName)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.rename(oldName, newName);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Rename an object.
     *
     * @param oldName
     *            the old name; may not be empty
     * @param newName
     *            the new name; may not be empty
     * @throws NamingException
     *             if a naming exception is encountered
     * @see javax.naming.NameException
     */
    public void rename(final String oldName, final String newName)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.rename(oldName, newName);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Unbinds the named object.
     *
     * @param name
     *            the name to unbind; may not be empty
     * @throws NamingException
     *             if a naming exception is encountered
     * @see javax.naming.NameException
     */
    public void unbind(final Name name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.unbind(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    /**
     * Unbinds the named object.
     *
     * @param name
     *            the name to unbind; may not be empty
     * @throws NamingException
     *             if a naming exception is encountered
     * @see javax.naming.NameException
     */
    public void unbind(final String name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.unbind(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public Object removeFromEnvironment(final String propName)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.removeFromEnvironment(propName);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public void close() throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.close();
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }

    }

    public Name composeName(final Name name, final Name prefix)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.composeName(name, prefix);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public String composeName(final String name, final String prefix)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.composeName(name, prefix);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public Context createSubcontext(final Name name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.createSubcontext(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public Context createSubcontext(final String name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.createSubcontext(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public void destroySubcontext(final Name name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.destroySubcontext(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }

    }

    public void destroySubcontext(final String name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            wrapped.destroySubcontext(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public Hashtable<?, ?> getEnvironment() throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.getEnvironment();
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public String getNameInNamespace() throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.getNameInNamespace();
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public NameParser getNameParser(final Name name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.getNameParser(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public NameParser getNameParser(final String name) throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.getNameParser(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public NamingEnumeration<NameClassPair> list(final Name name)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.list(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public NamingEnumeration<NameClassPair> list(final String name)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.list(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public NamingEnumeration<Binding> listBindings(final Name name)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.listBindings(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

    public NamingEnumeration<Binding> listBindings(final String name)
            throws NamingException {
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            return wrapped.listBindings(name);
        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }
    }

}
