/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: SmartContextFactory.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.smart.spi;

import java.util.Hashtable;
import java.util.List;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.naming.spi.InitialContextFactory;

import org.ow2.carol.cmi.smart.client.SmartClassLoader;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Initial Context factory used on the client side.
 * @author The new CMI team
 *
 */
public class SmartContextFactory implements InitialContextFactory {
    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(SmartContextFactory.class);

    /**
     * Property name for defining the protocol.
     */
    public static final String WRAPPED_PROTOCOL = "cmi.context.wrapped.protocol";

    /**
     * Property name for defining the initial provider URLs.
     */
    public static final String INITIAL_PROVIDER_URLS = "cmi.context.provider.urls";

    /**
     * Property name for defining the wrapped InitialContextFactory.
     */
    public static final String WRAPPED_INITIAL_CONTEXT_FACTORY = "cmi.context.wrapped.factory";

    /**
     * CMI initial context factory.
     */
    public static final String CMI_FACTORY = "org.ow2.carol.cmi.jndi.context.CMIInitialContextFactory";

    /**
     * Smart CMI configuration.
     */
    public static final String CMI_CONFIG_SMART = "cmi.config.smart";

    /**
     * Default URL.
     */
    private static final String DEFAULT_URL = "smart://localhost:2505";

    /**
     * Wrapped factory.
     */
    private static String wrappedFactoryName = null;

    /**
     * CMI delegating factory.
     */
    public static final String CMI_SMART_FACTORY = "cmi.smart.delegate.factory";

    /**
     * Instance of classloader that is used to load class.<br>
     * This is this classloader which will download class.
     */
    private static SmartClassLoader classLoader = null;

    /**
     * Provider URL to use for connecting.
     */
    private static List<String> providerURLs = null;

    /**
     * Protocol used in the client side.
     */
    private static final String DEFAULT_PROTOCOL = "jrmp";

    /**
     * Default constructor.<br>
     */
    public SmartContextFactory() {
        System.setProperty(CMI_SMART_FACTORY, this.getClass().getName());
    }

    /**
     * Creates an Initial Context for beginning name resolution. Special
     * requirements of this context are supplied
     *
     * @param environment the given environment.
     * @return the context.
     * @throws NamingException if no context can be built.
     */
    @SuppressWarnings("unchecked")
    public Context getInitialContext(final Hashtable environment)
            throws NamingException {

        // get the smart classLoader
        if (classLoader == null) {
            String currentProviderURL = DEFAULT_URL;

            String host = getHost(currentProviderURL);
            int port = getPort(currentProviderURL);
            classLoader = new SmartClassLoader(host, port);
        }

        // download the name of factory
        if (wrappedFactoryName == null) {
            wrappedFactoryName = classLoader.getWrappedFactoryName(DEFAULT_PROTOCOL);
            logger.info("Got the wrapped factory '" + wrappedFactoryName + "'.");
        }

        // download the list of provider url
        if (providerURLs == null) {
            providerURLs = classLoader.getProviderURLs(DEFAULT_PROTOCOL);
            logger.info("Got a list of remote PROVIDER_URLs."+providerURLs);
        }

        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(classLoader);
        try {
            // set the right factory
            environment.put(CMI_CONFIG_SMART, true);
            environment.put(Context.INITIAL_CONTEXT_FACTORY, CMI_FACTORY);
            environment.put(WRAPPED_INITIAL_CONTEXT_FACTORY, wrappedFactoryName);
            environment.put(INITIAL_PROVIDER_URLS, providerURLs);
            environment.put(WRAPPED_PROTOCOL, DEFAULT_PROTOCOL);

//            environment.put("cmi.mode.server", "true");
//            environment.put("cmi.server.start.replication", "true");
//            environment
//                    .put(
//                            "cmi.server.manager.class",
//                            "org.ow2.carol.cmi.controller.server.replication.jgroups.JGroupsClusterViewManager");
//            environment.put("cmi.server.manager.jgroups.conf", "jgroups-cmi.xml");
//            environment.put("cmi.server.manager.jgroups.gc", "10000");
//            environment.put("cmi.server.admin.domain", "CMI");
//            environment.put("cmi.server.admin.mbean", "CMIAdmin");
//            environment.put("cmi.server.provider.bind", "true");

            logger.info("The smart context is created!");
            // return wrapped context
            return new SmartContext(new InitialContext(environment),
                    classLoader);
//            return new SmartContext();

        } finally {
            Thread.currentThread().setContextClassLoader(cl);
        }

    }

    /**
     * Get the hostname of given url.
     * @param url the given url.
     * @return the correspond host name.
     */
    public String getHost(final String url) {
        String[] compos = url.split(":");
        String parthost = compos[1];
        String[] partcompos = parthost.split("/");
        String host = partcompos[partcompos.length - 1];
        return host;
    }

    /**
     * Get the port number of given url.
     * @param url the given url.
     * @return the correspond port number.
     */
    public int getPort(final String url) {
        String[] compos = url.split(":");
        String port = compos[compos.length - 1];
        int i = port.indexOf(",");
        if (i > 0) {
            String[] portcompos = port.split(",");
            port = portcompos[0];
        }
        return (new Integer(port).intValue());
    }

}
