
/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: ClassAnswer.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.smart.message;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;

import org.ow2.carol.cmi.smart.api.Message;
import org.ow2.carol.cmi.smart.api.Operations;

/**
 * Send a class to the client.
 * @author The new CMI team
 * @author Florent Benoit
 */

public class ClassAnswer implements Message {

    /**
     * Class name contained in the message.
     */
    private String className = null;

    /**
     * Array of bytes stored in the message.
     */
    private byte[] bytes = null;

    /**
     * Builds a new answer with the given class name and the array of bytes.
     * @param className the given name
     * @param bytes the array of bytes to store.
     */
    public ClassAnswer(final String className, final byte[] bytes) {
        this.className = className;
        this.bytes = bytes;
    }

    /**
     * Builds a new answer with the given bytebuffer.
     * @param dataBuffer buffer containing the data to extract.
     */
    public ClassAnswer(final ByteBuffer dataBuffer) {

        // Get length of the class name
        int nameLength = dataBuffer.getInt();

        // allocate buffer with the size of the name
        ByteBuffer nameBuffer = ByteBuffer.allocate(nameLength);
        for (int l = 0; l < nameLength; l++) {
            byte b = dataBuffer.get();
            nameBuffer.put(b);
        }

        // decode the name
        nameBuffer.position(0);

        // set it
        this.className = decode(nameBuffer);

        // rest of bytes = bytecode
        this.bytes = new byte[dataBuffer.limit() - dataBuffer.position()];
        int k = 0;
        for (int i = dataBuffer.position(); i < dataBuffer.limit(); i++) {
            this.bytes[k++] = dataBuffer.get(i);
        }
    }

    /**
     * Gets a message to send.
     * @return the bytebuffer to send
     */
    public ByteBuffer getMessage() {
        ByteBuffer message = getMessageContent();

        // compute length
        int length = HEADER_SIZE;
        if (message != null) {
            length += message.capacity();
        }

        // Create ByteBuffer
        ByteBuffer byteBuffer = ByteBuffer.allocate(length);

        // Append header
        byteBuffer.put(getOperationCode());

        if(message != null){
        byteBuffer.putInt(message.capacity());

        // append inner message (go to position 0 first)
        message.position(0);
        byteBuffer.put(message);
        }

        // reset our position
        byteBuffer.position(0);

        // return buffer
        return byteBuffer;
    }

    /**
     * Gets the Operation Code of this answer.
     * @return the operation code.
     */
    public byte getOperationCode() {
        return Operations.CLASS_ANSWER;
    }

    /**
     * Gets the content of this message (only this part, not the header).
     * @return the content of this message.
     */
    public ByteBuffer getMessageContent() {
        // Encode the classname
        ByteBuffer nameBuffer = encode(className);
        nameBuffer.position(0);

        // create buffer : length's name(int) + name + bytecode
        ByteBuffer messageBuffer = ByteBuffer.allocate(INT_BYTE_SIZE + nameBuffer.capacity() + bytes.length);

        // appends length
        messageBuffer.putInt(nameBuffer.capacity());

        // Needs to append the name
        messageBuffer.put(nameBuffer);

        // Bytecode
        messageBuffer.put(bytes);

        return messageBuffer;
    }

    /**
     * Gets the name of this class.
     * @return the name of this class.
     */
    public String getClassName() {
        return this.className;
    }

    /**
     * Gets bytecode of the class.
     * @return te bytecode of the class
     */
    public byte[] getByteCode() {
        return this.bytes;
    }

    /**
     * Encode the given string into a bytebuffer.
     * @param str the given string
     * @return a bytebuffer with UTF-8 encoded string
     */
    protected ByteBuffer encode(final String str) {
        byte[] bytes = null;
        try {
            bytes = str.getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        ByteBuffer buffer = ByteBuffer.allocate(bytes.length);
        buffer.put(bytes);
        return buffer;

    }

    /**
     * Decode the string encoded in the bytebuffer in UTF-8 format.
     * @param buffer the given buffer to analyze.
     * @return the decoded string
     */
    protected String decode(final ByteBuffer buffer) {
        Charset charset = Charset.forName("UTF-8");
        CharsetDecoder charsetDecoder = charset.newDecoder();

        CharBuffer charBuffer = null;
        try {
            charBuffer = charsetDecoder.decode(buffer);
        } catch (CharacterCodingException e) {
           throw new IllegalStateException("Invalid characted encoding", e);
        }
        return charBuffer.toString();
    }

}
