
/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: carol@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 * --------------------------------------------------------------------------
 * $Id: ResourceRequest.java 1547 2007-12-13 21:32:55Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.carol.cmi.smart.message;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;

import org.ow2.carol.cmi.smart.api.Message;
import org.ow2.carol.cmi.smart.api.Operations;

/**
 * Client asks the resource.
 * @author The new CMI team
 * @author Florent Benoit
 */

public class ResourceRequest implements Message {

    /**
     * Resource name used for this message.
     */
    private String resourceName = null;


    /**
     * Builds a new message with the given name.
     * @param resourceName the given resource name.
     */
    public ResourceRequest(final String resourceName) {
        this.resourceName = resourceName;
    }

    /**
     * Builds a message by using the data contains in the bytebuffer.
     * @param dataBuffer the data of the message to extract.
     */
    public ResourceRequest(final ByteBuffer dataBuffer) {
        // rest of bytes = resourceName
        this.resourceName = decode(dataBuffer);
    }

    /**
     * Gets a message to send.
     * @return the bytebuffer to send
     */
    public ByteBuffer getMessage() {
        ByteBuffer message = getMessageContent();

        // compute length
        int length = HEADER_SIZE;
        if (message != null) {
            length += message.capacity();
        }

        // Create ByteBuffer
        ByteBuffer byteBuffer = ByteBuffer.allocate(length);

        // Append header
        byteBuffer.put(getOperationCode());
        byteBuffer.putInt(message.capacity());

        // append inner message (go to position 0 first)
        message.position(0);
        byteBuffer.put(message);

        // reset our position
        byteBuffer.position(0);

        // return buffer
        return byteBuffer;
    }

    /**
     * Gets the OperationCode of this message.
     * @return the operation code.
     */
    public byte getOperationCode(){
        return Operations.RESOURCE_REQUEST;
    }

    /**
     * Gets the content of this message (only this part, not the header).
     * @return the content of this message.
     */
    public ByteBuffer getMessageContent() {
        // Needs to send the name
        ByteBuffer nameBuffer = encode(resourceName);
        return nameBuffer;
    }

    /**
     * Gets the class name of this message.
     * @return the name of the class.
     */
    public String getResourceName() {
        return this.resourceName;
    }

    /**
     * Encode the given string into a bytebuffer.
     * @param str the given string
     * @return a bytebuffer with UTF-8 encoded string
     */
    protected ByteBuffer encode(final String str) {
        byte[] bytes = null;
        try {
            bytes = str.getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        ByteBuffer buffer = ByteBuffer.allocate(bytes.length);
        buffer.put(bytes);
        return buffer;
    }

    /**
     * Decode the string encoded in the bytebuffer in UTF-8 format.
     * @param buffer the given buffer to analyze.
     * @return the decoded string
     */
    protected String decode(final ByteBuffer buffer) {
        Charset charset = Charset.forName("UTF-8");
        CharsetDecoder charsetDecoder = charset.newDecoder();

        CharBuffer charBuffer = null;
        try {
            charBuffer = charsetDecoder.decode(buffer);
        } catch (CharacterCodingException e) {
           throw new IllegalStateException("Invalid characted encoding", e);
        }
        return charBuffer.toString();
    }
}
